<?php

namespace MSML\AMF\Middleware;

use Carbon\Carbon;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\RequestInterface;
use Propaganistas\LaravelPhone\PhoneNumber;

class AmfMiddleware
{
    /**
     * Maps the key names of the JSON data in the request body, and returns a callable middleware that can be used
     * in the "withMiddleware" of the HTTP client.
     */
    public static function mapRequest(): callable
    {
        return Middleware::mapRequest(function (RequestInterface $request) {
            if ($request->getMethod() === 'GET') {
                return $request;
            }

            $data = json_decode($request->getBody()->getContents(), true);

            return new Request(
                method: $request->getMethod(),
                uri: $request->getUri(),
                headers: $request->getHeaders(),
                body: json_encode(self::mapKeynames($data, true)),
            );
        });
    }

    /**
     * Maps the key names of the JSON data in the response body, and returns a callable middleware that can be used
     * in the "withMiddleware" of the HTTP client.
     */
    public static function mapResponse(): callable
    {
        return Middleware::mapResponse(function (Response $response) {
            $data = json_decode($response->getBody()->getContents(), true);

            // Return a 500 error when something is giving an error message.
            // Maybe change later when AMF gives back error messages without 200 status code.
            if (data_get($data, 'errorMessage', false) && $response->getStatusCode() === 200) {
                abort(500, $data['errorMessage']);
            }

            return new Response(
                status: $response->getStatusCode(),
                headers: $response->getHeaders(),
                body: json_encode(self::mapKeynames($data)),
                version: $response->getProtocolVersion(),
            );
        });
    }

    /**
     * Map AMF keynames back and forth to readable keynames.
     */
    private static function mapKeynames(array $data, bool $flip = false): array
    {
        // Map multiple arrays.
        if (count($data) > 0 && is_array($data[0] ?? false)) {
            return collect($data)->map(function ($d) use ($flip) {
                return self::mapData($d, $flip);
            })->toArray();
        }

        // Map single array.
        return self::mapData($data, $flip);
    }

    /**
     * Maps an array of data using a given mapping object.
     */
    private static function mapData(array $data, bool $flip): array
    {
        $phoneNumbers = collect(config('amf.mapping.phone_numbers'))->flip();
        $dates = collect(config('amf.mapping.dates'))->flip();
        $booleans = collect(config('amf.mapping.booleans'));
        $fields = collect(config('amf.mapping.fields'));
        $fields = $flip ? $fields->flip() : $fields;

        return collect($data)->mapWithKeys(function ($value, $key) use ($fields, $booleans, $flip, $phoneNumbers, $dates) {
            $originalKey = $flip ? $fields->get($key, $key) : $key;

            if (isset($phoneNumbers[$originalKey])) {
                $value = self::phoneNumbers(value: $value, flip: $flip);
            }

            if (isset($dates[$originalKey])) {
                $value = self::dates(value: $value, flip: $flip);
            }

            if (isset($booleans[$originalKey])) {
                $value = $flip
                    ? collect($booleans[$originalKey])->search((bool)$value)
                    : $booleans[$originalKey][$value] ?? false;
            }

            $key = $fields->get($key, $key);

            return [$key => $value];
        })->toArray();
    }

    /**
     * Format phone numbers to E164 format or strip "+" for AMF.
     */
    private static function phoneNumbers(string|null $value, bool $flip): string|null
    {
        if ($flip) {
            return str_replace('+', '', $value);
        }

        try {
            $phoneNumber = (new PhoneNumber($value, 'NL'))->formatE164();
        } catch (\Throwable $th) {
            $phoneNumber = null;
        }

        return $phoneNumber;
    }

    /**
     * Format dates to desired format for portal or AMF.
     */
    private static function dates(Carbon|int|string|null $value, bool $flip): false|int|string|null
    {
        if ($flip) {
            $value = gettype($value) === 'object' && (new \ReflectionClass($value))->implementsInterface('Carbon\CarbonInterface')
                ? $value
                : Carbon::make($value);

            return $value?->format('Ymd') ?? 0;
        }

        try {
            $date = Carbon::createFromFormat('Ymd', $value);
        } catch (\Throwable $th) {
            $date = 0;
        }

        return $date;
    }
}
