<?php

namespace MSML\BusinessCentralService\Services\Query\Traits;

use Closure;
use Illuminate\Http\Client\Response;
use MSML\Exceptions\BusinessCentralClientException;

trait UpdateDeleteOperations
{
    /**
     * The tag required to match the current object status.
     */
    private string|null $versionTag = null;

    /**
     * The callbacks to be performed if the request recieves a 409 Coflict response.
     */
    private array $versionMismatchCallbacks = [];

    /**
     * Register an If-Match header for update/delete operations.
     */
    public function withVersionTag(string $versionTag): self
    {
        $this->versionTag = $versionTag;

        return $this;
    }

    /**
     * Issue a PUT request to the given URL.
     */
    public function put(string $systemId, array $data): Response
    {
        return $this->call('put', $systemId, $data);
    }

    /**
     * Issue a PATCH request to the given URL.
     */
    public function patch(string $systemId, array $data): Response
    {
        return $this->call('patch', $systemId, $data);
    }

    /**
     * Issue a DELETE request to the given URL.
     */
    public function delete(string $systemId): Response
    {
        return $this->call('delete', $systemId);
    }

    /**
     * Issue a PUT, PATCH or DELETE request to the given URL.
     */
    private function call(string $method, string $systemId, array $data = []): Response
    {
        $this->endpoint = "{$this->endpoint}($systemId)";

        $client = $this->service->getClient();

        $client = $this->versionTag
            ? (clone $client)->withHeader('If-Match', $this->versionTag)
            : $client;

        $callArguments = $method === 'delete'
            ? [$this->endpoint]
            : [$this->endpoint, $data];

        try {
            $response = $client->withQueryParameters($this->queryParams)
                ->{$method}(...$callArguments);
        } catch (BusinessCentralClientException $e) {
            if ($e->httpStatusCode === 409) {
                $this->runVersionMismatchCallbacks();
            }

            throw ($e);
        }

        // Depending on the service configuration, a faulty response does
        // not necessairly result in an exception. Therefore we perform
        // a separate check whether the response is a version mismatch.
        if ($response->status() === 409) {
            $this->runVersionMismatchCallbacks();
        }

        return $response;
    }

    /**
     * Specify an action that should be performed if the request recieves a 409 Coflict response.
     */
    public function onVersionMismatch(Closure $callback): self
    {
        $this->versionMismatchCallbacks[] = $callback;

        return $this;
    }

    /**
     * Execute the callbacks in case of a 409 Coflict response.
     */
    public function runVersionMismatchCallbacks(): void
    {
        foreach ($this->versionMismatchCallbacks as $callback) {
            $callback();
        }
    }
}
