<?php

namespace MSML\BusinessCentralService\Services;

use Closure;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\RequestException;
use MSML\BusinessCentralService\Services\Traits\ChecksConfig;
use MSML\BusinessCentralService\Middleware\BusinessCentralMiddleware;
use MSML\BusinessCentralService\Exceptions\BusinessCentralClientException;
use MSML\BusinessCentralService\Services\Query\BusinessCentralQueryBuilder;

class BusinessCentralService
{
    use ChecksConfig;

    /**
     * The client instance.
     */
    private PendingRequest $client;

    /**
     * Create a new service instance.
     */
    public function __construct(public array $options = [])
    {
        $this->ensureNoConfigValuesAreMissing('business-central-service.credentials');

        $this->buildClient();
    }

    /**
     * (Re)build the pending request based on the stored options.
     */
    private function buildClient(): void
    {
        $log = data_get($this->options, 'log', true);
        $throw = data_get($this->options, 'throw', true);
        $report = data_get($this->options, 'report', true);
        $pathKey = data_get($this->options, 'path', 'default');
        $path = config("business-central-service.paths.$pathKey");

        $baseUrl = config('business-central-service.credentials.base_url') . $path;
        $company = config('business-central-service.credentials.company');

        // Since we're working with OAuth2 bearer tokens that may expire, we have
        // to make sure that our custom middleware always skips 401 responses
        // and allows for the retry logic to generate a new Bearer token.
        $excludedResponseCodes = Arr::wrap(data_get($this->options, 'excludedResponseCodes', []));
        $excludedResponseCodes = array_unique([...$excludedResponseCodes, 401]);

        $clientMiddleware = new BusinessCentralMiddleware([
            'log_channel'           => 'business-central-errors',
            'excludedResponseCodes' => $excludedResponseCodes,
            'log'                   => $log,
            'throw'                 => $throw ? BusinessCentralClientException::class : false,
            'report'                => $report,
        ]);

        $credentials = $this->getCredentials();

        $this->client = Http::acceptJson()
            ->withToken($credentials->get('access_token'))
            ->withMiddleware($clientMiddleware->logErrors())
            ->baseUrl($baseUrl)
            ->retry(times: 3, sleepMilliseconds: 3000, when: $this->retryLogic(), throw: false)
            ->withQueryParameters(['company' => $company]);
    }

    /**
     * Get the required OAuth2 credentials.
     */
    public function getCredentials(bool $force = false): Collection
    {
        if ($force) {
            Cache::tags('auth')->forget('business_central');
        }

        $bearerCacheTtl = config('business-central-service.bearer_token_cache_ttl');
        $bearerCacheTtlSeconds = \Carbon\CarbonInterval::fromString($bearerCacheTtl)->totalSeconds;

        return Cache::tags('auth')->remember('business_central', $bearerCacheTtlSeconds, function () {
            $host = data_get(parse_url(config('business-central-service.credentials.auth_url')), 'host');

            $authUrl = Str::replace(
                '%tenantId%',
                config('business-central-service.credentials.tenant_id'),
                config('business-central-service.credentials.auth_url')
            );

            $response = Http::withHeaders(['Host' => $host])
                ->asForm()
                ->post($authUrl, [
                    'grant_type'    => config('business-central-service.credentials.grant_type'),
                    'scope'         => config('business-central-service.credentials.scope'),
                    'client_id'     => config('business-central-service.credentials.client_id'),
                    'client_secret' => config('business-central-service.credentials.client_secret'),
                ]);

            return $response->collect()->only(['access_token']);
        });
    }

    /**
     * Define the retry logic for failed HTTP requests.
     */
    private function retryLogic(): Closure
    {
        return function (Exception $exception, PendingRequest $request) {
            if (!$exception instanceof RequestException || $exception->response->status() !== 401) {
                return false;
            }

            $newToken = $this->getCredentials(force: true)->get('access_token');
            $request->withToken($newToken);

            return true;
        };
    }

    /**
     * Return the HTTP client.
     */
    public function getClient(): PendingRequest
    {
        return $this->client;
    }

    /**
     * Update the service options with a given options array.
     */
    public function setOptions(array $options): self
    {
        $this->options = [
            ...$this->options,
            ...$options,
        ];

        $this->buildClient();

        return $this;
    }

    /**
     * Update the service options with a single given option.
     */
    public function setOption(string $key, mixed $value): self
    {
        $this->setOptions([$key => $value]);
        $this->buildClient();

        return $this;
    }

    /**
     * Initialize a query by selecting and endpoint.
     */
    public function endpoint(string $endpoint): BusinessCentralQueryBuilder
    {
        return new BusinessCentralQueryBuilder($this, $endpoint);
    }

    /**
     * Initialize a query without an explicit endpoint selected.
     */
    public function builder(): BusinessCentralQueryBuilder
    {
        return new BusinessCentralQueryBuilder($this);
    }
}
