<?php

namespace MSML\BusinessCentralService\Services\Query\Traits;

use Closure;
use stdClass;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use InvalidArgumentException;
use Illuminate\Support\Collection;
use MSML\BusinessCentralService\Services\Query\BusinessCentralQueryBuilder;

trait GetOperations
{
    /**
     * The translation of familiar operator's to BC-style.
     */
    private array $operatorMapping = [
        '='  => 'eq',
        '!=' => 'ne',
        '>'  => 'gt',
        '>=' => 'ge',
        '<'  => 'lt',
        '<=' => 'le',
    ];

    /**
     * The contains constraints for the query.
     */
    private array $containsConstraints = [];

    /**
     * The where constraints for the query.
     */
    private array $wheres = [];

    /**
     * The orderings for the query.
     */
    private array $orders = [];

    /**
     * Determine whether the end result shoule be converted to a collection.
     */
    private bool $asCollection = false;

    /**
     * The startsWith constraints for the query.
     */
    private array $startsWiths = [];

    /**
     * The operator constants.
     */
    private const CONTAINS = 'contains';

    private const WHERE = 'where';

    private const OR_WHERE = 'orWhere';

    private const STARTS_WITH = 'startsWith';

    /**
     * Convert the query end result to a collection.
     */
    public function asCollection(): self
    {
        $this->asCollection = true;

        return $this;
    }

    /**
     * Set the columns to be selected.
     */
    public function select(mixed $columns): self
    {
        $columns = is_array($columns) ? $columns : func_get_args();
        $columns = implode(',', $columns);
        data_set($this->queryParams, '$select', $columns, overwrite: true);

        return $this;
    }

    /**
     * Set the "limit" value of the query.
     */
    public function limit(int $limit): self
    {
        data_set($this->queryParams, '$top', $limit, overwrite: true);

        return $this;
    }

    /**
     * Set the "offset" value of the query.
     */
    public function offset(int $offset): self
    {
        data_set($this->queryParams, '$skip', $offset, overwrite: true);

        return $this;
    }

    /**
     * Set the relationships that should be loaded along with the query.
     */
    public function with(mixed $relations): self
    {
        $relations = is_array($relations) ? $relations : func_get_args();
        $relations = implode(',', $relations);
        data_set($this->queryParams, '$expand', $relations, overwrite: true);

        return $this;
    }

    /**
     * Add contains to the query.
     */
    public function contains(Closure|string $column, string $value = '', string $boolean = 'and'): self
    {
        $this->containsConstraints[] = [
            'column'   => $column,
            'operator' => self::CONTAINS,
            'value'    => $value,
            'boolean'  => $boolean,
        ];

        return $this;
    }

    /**
     * Add contains to the query.
     */
    public function startsWith(Closure|string $column, string $value = '', string $boolean = 'and'): self
    {
        $this->containsConstraints[] = [
            'column'   => $column,
            'operator' => self::STARTS_WITH,
            'value'    => $value,
            'boolean'  => $boolean,
        ];

        return $this;
    }

    /**
     * Add a where clause to the query.
     */
    public function where(Closure|string $column, mixed $operator = null, mixed $value = new stdClass, string $boolean = 'and'): self
    {
        if ($column instanceof Closure) {
            return $this->whereNested($column, $boolean);
        }

        // Distinguish between a value that is 'null' or a value that has
        // not been provided at all by using a fallback object type.
        if (gettype($value) === 'object' && !($value instanceof Carbon)) {
            [$value, $operator] = [$operator, '='];
        }

        if (!in_array($operator, array_keys($this->operatorMapping))) {
            throw new InvalidArgumentException('Illegal operator and value combination.');
        }

        $this->wheres[] = compact(
            'column',
            'operator',
            'value',
            'boolean'
        );

        return $this;
    }

    /**
     * Add an "or where" clause to the query.
     */
    public function orWhere(Closure|string $column, string|null $operator = null, mixed $value = new stdClass): self
    {
        return $this->where($column, $operator, $value, 'or');
    }

    /**
     * Add a "where null" clause to the query.
     */
    public function whereNull(string $column): self
    {
        return $this->where($column, '=', null);
    }

    /**
     * Add a "where not null" clause to the query.
     */
    public function whereNotNull(string $column): self
    {
        return $this->where($column, '!=', null);
    }

    /**
     * Add a "where in" clause to the query.
     */
    public function whereIn(string $column, array $values, bool $inverse = false): self
    {
        $query = new BusinessCentralQueryBuilder;

        collect($values)->each(function ($value) use (&$query, $column, $inverse) {
            $operator = $inverse ? '!=' : '=';
            $method = $inverse ? self::WHERE : self::OR_WHERE;

            $query->{$method}($column, $operator, $value);
        });

        $this->wheres[] = [
            'query'   => $query,
            'boolean' => 'and',
        ];

        return $this;
    }

    /**
     * Add a "where not in" clause to the query.
     */
    public function whereNotIn(string $column, array $values): self
    {
        return $this->whereIn($column, $values, inverse: true);
    }

    /**
     * Add a "where between" clause to the query.
     */
    public function whereBetween(string $column, iterable $values): self
    {
        if ($values instanceof CarbonPeriod) {
            $values = [$values->start, $values->end];
        }

        if (count($values) !== 2) {
            throw new InvalidArgumentException('whereBetween only accepts a range of two values.');
        }

        return $this->where(function ($query) use ($column, $values) {
            return $query->where($column, '>=', $values[0])
                ->where($column, '<=', $values[1]);
        });
    }

    /**
     * Add an "order by" clause to the query.
     */
    public function orderBy(string $column, string $direction = 'asc'): self
    {
        if (!in_array($direction, ['asc', 'desc'], strict: true)) {
            throw new InvalidArgumentException('Order direction must be "asc" or "desc".');
        }

        $this->orders[$column] = $direction;

        return $this;
    }

    /**
     * Add a nested where statement to the query.
     */
    private function whereNested(Closure $callback, $boolean = 'and'): self
    {
        $query = $callback(new BusinessCentralQueryBuilder);

        $this->wheres[] = compact('query', 'boolean');

        return $this;
    }

    /**
     * Compile the where constraints into BC-friendly filter query string.
     */
    private function getFilterString(): string|null
    {
        return collect([...$this->wheres, ...$this->containsConstraints])
            ->reduce(function ($carry, $condition) {
                $operator = data_get($this->operatorMapping, data_get($condition, 'operator'));

                $value = data_get($condition, 'value');
                $parsedValue = $this->getParsedFilterValue($value);

                // Sperate query function for Business Central function queries.
                $query = $this->buildFunctionQuery($condition, $parsedValue);

                $whereCondition = array_key_exists('query', $condition)
                    ? "({$condition['query']->getFilterString()})"
                    : $query ?? "{$condition['column']} {$operator} {$parsedValue}";

                return $carry
                    ? "{$carry} {$condition['boolean']} {$whereCondition}"
                    : $whereCondition;
            }, null);
    }

    /**
     * Build query functions for BC startsWith filter.
     */
    private function buildFunctionQuery(array $condition, string|null $parsedValue = null): string|null
    {
        $operatorFunction = data_get($condition, 'operator');

        if (method_exists($this, $operatorFunction)) {
            return "{$operatorFunction}({$condition['column']}, {$parsedValue})";
        }

        return null;
    }

    /**
     * Compile the order constraints into BC-friendly order query string.
     */
    public function getOrderByString(): string|null
    {
        $orderByString = collect($this->orders)
            ->map(fn ($direction, $column) => "{$column} {$direction}")
            ->join(', ');

        return !empty($orderByString) ? $orderByString : null;
    }

    /**
     * Finalize and perform the HTTP GET call.
     */
    public function get(): array|Collection
    {
        $this->queryParams['$filter'] = $this->getFilterString();
        $this->queryParams['$orderby'] = $this->getOrderByString();

        $this->queryParams = array_filter($this->queryParams);

        $response = $this->service
            ->getClient()
            ->get($this->endpoint, $this->queryParams);

        $data = $response->collect('value');

        if ($this->castValues) {
            $data = $this->processCasts($data);
        }

        return $this->asCollection
            ? $data
            : $data->toArray();
    }

    /**
     * Finalize and perform the HTTP GET call with the result as a collection.
     */
    public function collect(): Collection
    {
        return $this->asCollection()
            ->get();
    }

    /**
     * Get the first item from the GET call results.
     */
    public function first(): array|Collection|null
    {
        $response = $this->limit(1)
            ->get();

        $data = $response[0] ?? null;

        return $this->asCollection ? collect($data) : $data;
    }

    /**
     * Add a basic where clause to the query, and return the first result.
     */
    public function firstWhere(...$arguments): array|Collection|null
    {
        $response = $this->limit(1)
            ->where(...$arguments)
            ->get();

        $data = $response[0] ?? null;

        return $this->asCollection ? collect($data) : $data;
    }

    /**
     * Retrieve the "count" result of the query.
     */
    public function count(): int
    {
        $this->endpoint = $this->endpoint . '/$count';

        data_set($this->queryParams, '$filter', $this->getFilterString(), overwrite: true);

        $response = $this->service
            ->getClient()
            ->get($this->endpoint, $this->queryParams);

        // Filter unwanted characters out.
        preg_match('/\d+/', $response->body(), $matches);

        return (int)data_get($matches, 0);
    }

    /**
     * Get a the filter value in the BC API-friendly format.
     */
    private function getParsedFilterValue(mixed $value): string
    {
        if (is_bool($value)) {
            return $value ? 'true' : 'false';
        }

        if (in_array(gettype($value), ['integer', 'double'])) {
            return (string)$value;
        }

        if ($value instanceof Carbon) {
            return $value->toBusinessCentralFormat();
        }

        return "'{$value}'";
    }
}
