<?php

namespace MSML\HttpClient\Middleware;

use Throwable;
use GuzzleHttp\Psr7\Stream;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use MSML\HttpClient\Accessories\XML;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use MSML\HttpClient\Exceptions\HttpClientException;

class HttpClientMiddleware
{
    public function __construct(protected array|null $options = [])
    {
        //
    }

    /**
     * Log failed HTTP requests.
     */
    public function logErrors(): callable
    {
        return function (callable $handler) {
            return function (RequestInterface $request, array $options) use ($handler) {
                $promise = $handler($request, $options);

                return $promise->then(function (ResponseInterface $response) use ($request) {
                    $successResponse = (bool)preg_match('/^2\d{2}$/', $response->getStatusCode());
                    $excludedResponseCodes = data_get($this->options, 'excludedResponseCodes', []);

                    if ($successResponse || in_array($response->getStatusCode(), $excludedResponseCodes)) {
                        return $response;
                    }

                    $defaultLog = config('logging.default', 'stack');

                    $logChannel = data_get($this->options, 'log_channel', $defaultLog);
                    $logMessage = $this->getLogMessage($request, $response);
                    $logData = $this->getLogData($request, $response);

                    if (data_get($this->options, 'log', true)) {
                        Log::channel($logChannel)
                            ->error($logMessage, $logData);
                    }

                    $this->handleException($logMessage, $logData);

                    return $response;
                });
            };
        };
    }

    /**
     * Form the request side of the log message.
     */
    protected function getRequestLog(RequestInterface $request): array
    {
        parse_str($request->getUri()->getQuery(), $parameters);

        return [
            'method'  => $request->getMethod(),
            'headers' => Arr::except($request->getHeaders(), 'Authorization'),
            'uri'     => [
                'scheme'      => $request->getUri()->getScheme(),
                'host'        => $request->getUri()->getHost(),
                'path'        => $request->getUri()->getPath(),
                'query'       => $parameters,
                'composedUri' => (string)$request->getUri(),
            ],
            'body' => $this->parseStream($request->getBody(), true),
        ];
    }

    /**
     * Form the response side of the log message.
     */
    protected function getResponseLog(ResponseInterface $response): array
    {
        return [
            'reasonPhrase' => $response->getReasonPhrase(),
            'statusCode'   => $response->getStatusCode(),
            'headers'      => $response->getHeaders(),
            'body'         => $this->parseStream($response->getBody(), true),
        ];
    }

    /**
     * Get the formatted request and response log data.
     */
    protected function getLogData(RequestInterface $request, ResponseInterface $response): array
    {
        return [
            'request'  => $this->getRequestLog($request),
            'response' => $this->getResponseLog($response),
        ];
    }

    /**
     * Get the log message.
     */
    protected function getLogMessage(RequestInterface $request, ResponseInterface $response): string
    {
        return data_get($this->options, 'log_message', 'Outgoing HTTP request error');
    }

    /**
     * Parse the response body to a readable format.
     */
    protected function parseStream(Stream $stream): array|string|null
    {
        if (($jsonArray = json_decode($stream, true)) !== null) {
            return $jsonArray;
        }

        if (($xmlArray = XML::toArray(xmlString: $stream, throw: false)) !== null) {
            return $xmlArray;
        }

        return $stream;
    }

    /**
     * Throw a possibly provided exception.
     */
    protected function handleException(string $logMessage, array $logData): void
    {
        $exception = data_get($this->options, 'throw');
        $report = data_get($this->options, 'report', true);

        $throwInitiatedException = $exception instanceof Throwable;
        $throwGenericException = gettype($exception) === 'boolean' && $exception;
        $throwCustomException = gettype($exception) === 'string'
            && class_exists($exception)
            && isset(class_implements($exception)['Throwable']);

        $exception = match (true) {
            $throwInitiatedException => $exception,
            $throwGenericException   => new HttpClientException($logMessage),
            $throwCustomException    => new $exception($logMessage),
            default                  => null
        };

        if (!$exception) {
            return;
        }

        if (property_exists($exception, 'httpStatusCode')) {
            $exception->httpStatusCode = data_get($logData, 'response.statusCode');
        }

        if (!$report && method_exists($exception, 'dontReport')) {
            $exception = $exception->dontReport();
        }

        if (property_exists($exception, 'httpStatusCode')) {
            $exception->httpStatusCode = $statusCode = data_get($logData, 'response.statusCode');
        }

        if (method_exists($exception, 'setContext')) {
            $exception->setContext(name: "Error {$statusCode}: {$logMessage}", data: $logData);
        }

        throw $exception;
    }
}
