<?php

declare(strict_types=1);

namespace MSML\AppVersion\Models;

use Illuminate\Support\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use MSML\AppVersion\Enums\MobileAppTypeEnum;
use Illuminate\Database\Eloquent\Attributes\Scope;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use MSML\AppVersion\Enums\MobileAppVersionPlatformEnum;

/**
 * @property int $id
 * @property string $version
 * @property string $text
 * @property MobileAppVersionPlatformEnum $platform
 * @property int|null $user_id
 * @property string $url
 * @property bool $force_update
 * @property array $properties
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $published_at
 * @property-read MobileApp|Model $mobileApp
 *
 * @method static Builder<MobileAppVersion> platform(MobileAppVersionPlatformEnum $platform)
 * @method static Builder<MobileAppVersion> version(string $version)
 * @method static Builder<MobileAppVersion> versionLike(string $needle)
 * @method static Builder<MobileAppVersion> createdBetween(mixed $from = null, mixed $to = null)
 * @method static Builder<MobileAppVersion> ofAppType(Builder $query, MobileAppTypeEnum $type)
 */
class MobileAppVersion extends Model
{
    /**
     * @var list<string>
     */
    protected $guarded = [];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'platform'     => MobileAppVersionPlatformEnum::class,
            'force_update' => 'boolean',
            'published_at' => 'datetime',
            'properties'   => 'array',
        ];
    }

    /**
     * @return BelongsTo<MobileApp|Model, $this>
     */
    public function mobileApp(): BelongsTo
    {
        return $this->belongsTo(config('app-version.models.mobile_app'));
    }

    /**
     * @return BelongsTo<Model, $this>
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(config('auth.providers.users.model'));
    }

    #[Scope]
    protected function platform(Builder $query, MobileAppVersionPlatformEnum $platform): void
    {
        $query->where('platform', $platform->value);
    }

    #[Scope]
    protected function version(Builder $query, string $version): void
    {
        $query->where('version', $version);
    }

    #[Scope]
    protected function versionLike(Builder $query, string $needle): void
    {
        $query->whereRaw('LOWER(version) LIKE ?', ['%' . mb_strtolower($needle) . '%']);
    }

    #[Scope]
    protected function createdBetween(Builder $query, mixed $from = null, mixed $to = null): void
    {
        $query
            ->when($from, fn (Builder $q) => $q->where('created_at', '>=', Carbon::parse($from)->startOfDay()))
            ->when($to, fn (Builder $q) => $q->where('created_at', '<=', Carbon::parse($to)->endOfDay()));
    }

    #[Scope]
    protected function ofAppType(Builder $query, MobileAppTypeEnum $type): void
    {
        $query->whereHas('mobileApp', fn (Builder $q) => $q->where('type', $type->value));
    }
}
