<?php

declare(strict_types=1);

namespace MSML\AppVersion\Models;

use Illuminate\Support\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use MSML\AppVersion\Enums\AppVersionPlatformEnum;
use Illuminate\Database\Eloquent\Attributes\Scope;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use MSML\AppVersion\Models\App;

/**
 * @property int $id
 * @property string $version
 * @property string $text
 * @property AppVersionPlatformEnum $platform
 * @property int|null $user_id
 * @property string $url
 * @property bool $force_update
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $published_at
 * 
 * @method static Builder<AppVersion> platform(AppVersionPlatformEnum $platform)
 * @method static Builder<AppVersion> version(string $version)
 * @method static Builder<AppVersion> versionLike(string $needle)
 * @method static Builder<AppVersion> createdBetween(mixed $from = null, mixed $to = null)
 * @method static \MSML\AppVersion\Database\Factories\AppVersionFactory factory()
 */
class AppVersion extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * @var list<string>
     */
    protected $guarded = [];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'platform'     => AppVersionPlatformEnum::class,
            'force_update' => 'boolean',
            'published_at' => 'datetime',
        ];
    }

    /**
     * @return BelongsTo<App, $this>
     */
    public function app(): BelongsTo
    {
        return $this->belongsTo(App::class);
    }

    /**
     * @return BelongsTo<Model, $this>
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(config('auth.providers.users.model'));
    }

    #[Scope]
    protected function platform(Builder $query, AppVersionPlatformEnum $platform): void
    {
        $query->where('platform', $platform->value);
    }

    #[Scope]
    protected function version(Builder $query, string $version): void
    {
        $query->where('version', $version);
    }

    #[Scope]
    protected function versionLike(Builder $query, string $needle): void
    {
        $query->whereRaw('LOWER(version) LIKE ?', ['%' . mb_strtolower($needle) . '%']);
    }

    #[Scope]
    protected function createdBetween(Builder $query, mixed $from = null, mixed $to = null): void
    {
        $query
            ->when($from, fn(Builder $q) => $q->where('created_at', '>=', Carbon::parse($from)->startOfDay()))
            ->when($to, fn(Builder $q) => $q->where('created_at', '<=', Carbon::parse($to)->endOfDay()));
    }
}
