<?php

declare(strict_types=1);

namespace MSML\AppVersion\Actions;

class CheckAppVersion
{
    /**
     * @return array<string, string|null|bool>
     */
    public function handle(string $identifier, string $platform, string $version): array
    {
        $modelAppVersion = config('app-version.models.app_version');
        $modelApp = config('app-version.models.app');

        $allPublished = $modelAppVersion::query()->whereRelation('app', 'identifier', $identifier)
            ->where('platform', $platform)
            ->whereNotNull('published_at')
            ->get();

        if ($allPublished->isEmpty()) {
            $appExists = $modelApp::query()->where('identifier', $identifier)->exists();

            if (!$appExists) {
                return [
                    'allowed' => false,
                    'message' => config('app-version.messages.missing_app'),
                    'latest'  => null,
                    'url'     => null,
                ];
            }

            return [
                'allowed' => true,
                'message' => null,
                'latest'  => null,
                'url'     => null,
            ];
        }

        $required = $allPublished->reduce(function ($carry, $item) {
            if (!$carry) {
                return $item;
            }

            return version_compare($item->version, $carry->version, '>') ? $item : $carry;
        });

        $forceUpdateVersions = $allPublished->filter(fn ($item) => $item->force_update);

        $minRequired = $forceUpdateVersions->reduce(function ($carry, $item) {
            if (!$carry) {
                return $item;
            }

            return version_compare($item->version, $carry->version, '>') ? $item : $carry;
        });

        $allowed = !$minRequired || version_compare($version, $minRequired->version, '>=');

        $message = null;
        if (!$allowed) {
            $message = $minRequired->text ?? config('app-version.messages.update_required');
        }

        return [
            'allowed' => $allowed,
            'message' => $message,
            'latest'  => $required->version,
            'url'     => $required->url,
        ];
    }
}
