<?php

namespace MSML\AppVersion\Models;

use Illuminate\Support\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use MSML\AppVersion\Enums\AppVersionPlatformEnum;
use Illuminate\Database\Eloquent\Attributes\Scope;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * @property int $id
 * @property string $version
 * @property string $text
 * @property AppVersionPlatformEnum $platform
 * @property int|null $user_id
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 *
 * @method static Builder<AppVersion> platform(AppVersionPlatformEnum $platform)
 * @method static Builder<AppVersion> version(string $version)
 * @method static Builder<AppVersion> versionLike(string $needle)
 * @method static Builder<AppVersion> createdBetween(mixed $from = null, mixed $to = null)
 * @method static \MSML\AppVersion\Database\Factories\AppVersionFactory factory()
 */
class AppVersion extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = ['version', 'text', 'platform', 'user_id'];

    protected $casts = [
        'platform' => AppVersionPlatformEnum::class,
    ];

    public function getTable(): string
    {
        return config('app-version.table', 'app_versions');
    }

    public function user(): BelongsTo
    {
        /** @var class-string<\Illuminate\Database\Eloquent\Model> $userModel */
        $userModel = (string)config('auth.providers.users.model');

        return $this->belongsTo($userModel);
    }

    #[Scope]
    protected function platform(Builder $query, AppVersionPlatformEnum $platform): void
    {
        $query->where('platform', $platform->value);
    }

    #[Scope]
    protected function version(Builder $query, string $version): void
    {
        $query->where('version', $version);
    }

    #[Scope]
    protected function versionLike(Builder $query, string $needle): void
    {
        $query->whereRaw('LOWER(version) LIKE ?', ['%' . mb_strtolower($needle) . '%']);
    }

    #[Scope]
    protected function createdBetween(Builder $query, mixed $from = null, mixed $to = null): void
    {
        $query
            ->when($from, fn (Builder $q) => $q->where('created_at', '>=', Carbon::parse($from)->startOfDay()))
            ->when($to, fn (Builder $q) => $q->where('created_at', '<=', Carbon::parse($to)->endOfDay()));
    }
}
