<?php

namespace MSML\Blueprint\DSL\Schema;

use Spatie\LaravelData\Data;
use Illuminate\Validation\Rule;
use MSML\Blueprint\DSL\Nodes\FieldNode;
use MSML\Blueprint\DSL\Nodes\AutoCompleteNode;

final class BlueprintSchema extends Data
{
    /**
     * @param  array<string,FieldNode>  $components
     * @param  array<string,list<string>>  $rules
     * @param  list<TabSchema>  $layout
     * @param  array<string,mixed>  $defaults
     */
    public function __construct(
        public string $key,
        public int $version,
        public array $defaults,
        public array $components,
        public array $rules,
        public array $layout,
    ) {}

    /**
     * @return array<string,list<string>>
     */
    public function laravelRules(): array
    {
        return $this->rules;
    }

    /**
     * @return array<string,mixed>
     */
    public function defaultValues(): array
    {
        return $this->defaults;
    }

    /**
     * @param  array<string,mixed>  $relations
     * @return array<string,mixed>
     */
    public function rulesWithScopes(array $relations): array
    {
        $rules = $this->rules;

        foreach ($this->components as $fieldKey => $component) {
            if (!$component instanceof AutoCompleteNode) {
                continue;
            }

            $scope = $component->scope;
            if ($scope === null) {
                continue;
            }

            $exists = collect($rules[$fieldKey] ?? [])->first(
                fn ($r) => is_string($r) && str_starts_with($r, 'exists:')
            );

            if ($exists === null) {
                continue;
            }

            [$table, $column] = explode(',', substr($exists, 7));
            $parentField = $scope['dependsOn'];
            $param = $scope['param'];

            $parentValue = $relations["{$parentField}_id"] ?? null;

            if ($parentValue !== null && (is_string($parentValue) || is_int($parentValue))) {

                $rules[$fieldKey] = array_map(function ($r) use ($table, $column, $param, $parentValue) {

                    return (is_string($r) && str_starts_with($r, 'exists:'))
                        ? Rule::exists($table, $column)->where($param, $parentValue)
                        : $r;

                }, $rules[$fieldKey] ?? []);
            }
        }

        return $rules;
    }

    /**
     * @param  array<string,mixed>  $overrides
     */
    public function withDefaults(array $overrides): self
    {
        return new self(
            key: $this->key,
            version: $this->version,
            defaults: array_replace($this->defaults, $overrides),
            components: $this->components,
            rules: $this->rules,
            layout: $this->layout,
        );
    }
}
