<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use MSML\Blueprint\DSL\Nodes\AccordionNode;
use MSML\Blueprint\DSL\Nodes\AccordionItemNode;
use MSML\Blueprint\DSL\Contracts\ElementBuilder;
use MSML\Blueprint\DSL\Schema\AccordionElementSchema;

final class AccordionBuilder implements ElementBuilder
{
    private int $span = 12;

    /** @var array<AccordionItemNode> */
    private array $items = [];

    private bool $multiple = false;

    /** @var array<string>|null */
    private array|null $defaultOpen = null;

    private string $chevronPosition = 'right';

    private string|null $variant = null;

    private string|null $radius = null;

    public function item(string $title, string $content): self
    {
        $this->items[] = new AccordionItemNode($title, $content);

        return $this;
    }

    public function multiple(bool $multiple = true): self
    {
        $this->multiple = $multiple;

        return $this;
    }

    /**
     * @param  string|array<string>  $titles
     */
    public function defaultOpen(array|string $titles): self
    {
        $this->defaultOpen = is_array($titles) ? $titles : [$titles];

        return $this;
    }

    public function firstOpen(): self
    {
        if (count($this->items) > 0) {
            $this->defaultOpen = [$this->items[0]->title];
        }

        return $this;
    }

    public function chevronLeft(): self
    {
        $this->chevronPosition = 'left';

        return $this;
    }

    public function chevronRight(): self
    {
        $this->chevronPosition = 'right';

        return $this;
    }

    public function chevronPosition(string $position): self
    {
        $this->chevronPosition = $position;

        return $this;
    }

    public function variant(string|null $variant): self
    {
        $this->variant = $variant;

        return $this;
    }

    public function radius(string|null $radius): self
    {
        $this->radius = $radius;

        return $this;
    }

    public function span(int $columns): self
    {
        $this->span = max(1, min(12, $columns));

        return $this;
    }

    public function toAccordionElementSchema(): AccordionElementSchema
    {
        $node = new AccordionNode(
            span: $this->span,
            items: $this->items,
            multiple: $this->multiple,
            defaultOpen: $this->defaultOpen,
            chevronPosition: $this->chevronPosition,
            variant: $this->variant,
            radius: $this->radius,
        );

        return new AccordionElementSchema($node);
    }
}
