<?php

namespace MSML\Blueprint\DSL\Builders;

use Closure;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\FieldBuilderMethods;
use MSML\Blueprint\DSL\Schema\RowElementSchema;
use MSML\Blueprint\DSL\Contracts\ElementBuilder;
use MSML\Blueprint\DSL\Traits\Common\WithTitleSize;
use MSML\Blueprint\DSL\Schema\ComponentElementSchema;

final class RowBuilder implements ElementBuilder
{
    use FieldBuilderMethods;
    use WithTitleSize;

    private int $span = 12;

    /** @var array<RowBuilder|PanelBuilder|FieldsetBuilder|DividerBuilder|AccordionBuilder|ComponentElementSchema> */
    private array $elements = [];

    public function __construct(
        private BlueprintCompiler $root,
        private int $sectionIndex,
        private int $rowIndex,
        private string|null $title = null,
    ) {}

    /**
     * @param  string|Closure(RowBuilder): void|null  $titleOrFn
     * @param  Closure(RowBuilder): void|null  $fn
     */
    public function row(Closure|string|null $titleOrFn = null, Closure|null $fn = null): self
    {
        if ($titleOrFn instanceof Closure) {
            $title = null;
            $fn = $titleOrFn;
        } else {
            $title = $titleOrFn;
        }

        $nestedRow = new self($this->root, $this->sectionIndex, -1, $title);

        if ($fn !== null) {
            $fn($nestedRow);
        }

        $this->elements[] = $nestedRow;

        return $nestedRow;
    }

    /**
     * @param  string|Closure(PanelBuilder): void|null  $titleOrFn
     * @param  Closure(PanelBuilder): void|null  $fn
     */
    public function panel(Closure|string|null $titleOrFn = null, Closure|null $fn = null): PanelBuilder
    {
        if ($titleOrFn instanceof Closure) {
            $title = null;
            $fn = $titleOrFn;
        } else {
            $title = $titleOrFn;
        }

        $panel = new PanelBuilder($this->root, $this->sectionIndex, -1, $title);

        if ($fn !== null) {
            $fn($panel);
        }

        $this->elements[] = $panel;

        return $panel;
    }

    /**
     * @param  string|Closure(FieldsetBuilder): void|null  $legendOrFn
     * @param  Closure(FieldsetBuilder): void|null  $fn
     */
    public function fieldset(Closure|string|null $legendOrFn = null, Closure|null $fn = null): FieldsetBuilder
    {
        if ($legendOrFn instanceof Closure) {
            $legend = null;
            $fn = $legendOrFn;
        } else {
            $legend = $legendOrFn;
        }

        $fieldset = new FieldsetBuilder($this->root, $this->sectionIndex, -1, $legend);

        if ($fn !== null) {
            $fn($fieldset);
        }

        $this->elements[] = $fieldset;

        return $fieldset;
    }

    public function divider(): DividerBuilder
    {
        $divider = new DividerBuilder;
        $this->elements[] = $divider;

        return $divider;
    }

    /**
     * @param  Closure(AccordionBuilder): void  $fn
     */
    public function accordion(Closure $fn): AccordionBuilder
    {
        $accordion = new AccordionBuilder;
        $fn($accordion);
        $this->elements[] = $accordion;

        return $accordion;
    }

    public function span(int $columns): self
    {
        $this->span = max(1, min(12, $columns));

        return $this;
    }

    protected function getCompiler(): BlueprintCompiler
    {
        return $this->root;
    }

    protected function getSectionIndex(): int
    {
        return $this->sectionIndex;
    }

    protected function getRowIndex(): int
    {
        return $this->rowIndex;
    }

    public function registerComponent(string $key, int $span = 12): void
    {
        $this->elements[] = new ComponentElementSchema($key, $span);
    }

    public function updateComponentSpan(string $key, int $span): void
    {
        foreach ($this->elements as $index => $element) {
            if ($element instanceof ComponentElementSchema && $element->key === $key) {
                $this->elements[$index] = new ComponentElementSchema($key, $span);
                break;
            }
        }
    }

    public function toRowElementSchema(): RowElementSchema
    {
        $elementSchemas = [];

        foreach ($this->elements as $element) {
            if ($element instanceof self) {
                $elementSchemas[] = $element->toRowElementSchema();
            } elseif ($element instanceof PanelBuilder) {
                $elementSchemas[] = $element->toPanelElementSchema();
            } elseif ($element instanceof FieldsetBuilder) {
                $elementSchemas[] = $element->toFieldsetElementSchema();
            } elseif ($element instanceof DividerBuilder) {
                $elementSchemas[] = $element->toDividerElementSchema();
            } elseif ($element instanceof AccordionBuilder) {
                $elementSchemas[] = $element->toAccordionElementSchema();
            } elseif ($element instanceof ComponentElementSchema) {
                $elementSchemas[] = $element;
            }
        }

        return new RowElementSchema($elementSchemas, $this->span, $this->title, $this->getTitleSize());
    }

    public function getTitle(): string|null
    {
        return $this->title;
    }
}
