<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\Nodes\FileInputNode;
use MSML\Blueprint\DSL\Contracts\FieldBuilder;
use MSML\Blueprint\DSL\Traits\Common\WithHelp;
use MSML\Blueprint\DSL\Traits\Common\WithRules;
use MSML\Blueprint\DSL\Traits\Core\NodeUpdater;
use MSML\Blueprint\DSL\Traits\Core\SpanUpdater;
use MSML\Blueprint\DSL\Traits\Specialized\WithFile;
use MSML\Blueprint\DSL\Traits\Specialized\WithUpload;
use MSML\Blueprint\DSL\Traits\Core\CoreBuilderMethods;

final class FileInputBuilder implements FieldBuilder
{
    use CoreBuilderMethods;
    use NodeUpdater;
    use SpanUpdater;
    use WithFile;
    use WithHelp;
    use WithRules;
    use WithUpload;

    protected FileInputNode $node;

    public function __construct(
        BlueprintCompiler $root,
        int $sectionIndex,
        int $rowIndex,
        string $key,
        string|null $label,
        FieldsetBuilder|PanelBuilder|RowBuilder|TabBuilder $context
    ) {
        $this->root = $root;
        $this->sectionIndex = $sectionIndex;
        $this->rowIndex = $rowIndex;
        $this->key = $key;
        $this->context = $context;

        $this->node = new FileInputNode(
            label: $label,
            span: 12,
            accept: null,
            maxSize: null,
            help: null,
            uploadEndpoint: null,
            namespace: null,
            collection: null,
            allowBrowse: false,
            selected: null
        );

        $this->registerInLayout();
        $this->updateNode();
    }

    /** @param array<mixed> $value**/
    public function value(array|string $value): self
    {
        $this->root->defaults[$this->key] = $value;

        return $this;
    }

    public function selectedFrom(string $dataKey, string $idKey = 'id', string $urlKey = 'url', string $nameKey = 'name'): self
    {
        $source = $this->normalizeDataSource();

        if ($source === null) {
            return $this;
        }

        $payload = $this->valueFromPath($source, $dataKey);

        if (!is_array($payload)) {
            return $this;
        }

        $id = $this->valueFromPath($payload, $idKey);
        $url = $this->valueFromPath($payload, $urlKey);
        $name = $this->valueFromPath($payload, $nameKey);

        if ($id === null || $url === null || $name === null) {
            return $this;
        }

        $this->root->defaults[$this->key] = $id;
        $this->updateNodeWith([
            'selected' => [
                'id'   => $id,
                'url'  => $url,
                'name' => $name,
            ],
        ]);

        return $this;
    }

    /**
     * @param  array<mixed>  $source
     */
    private function valueFromPath(array $source, string $path): mixed
    {
        if ($path === '') {
            return null;
        }

        $segments = explode('.', $path);
        $value = $source;

        foreach ($segments as $segment) {
            if (!is_array($value) || !array_key_exists($segment, $value)) {
                return null;
            }

            $value = $value[$segment];
        }

        return $value;
    }

    /**
     * @return array<mixed>|null
     */
    private function normalizeDataSource(): array|null
    {
        $data = $this->root->data;

        if ($data instanceof \Spatie\LaravelData\Data) {
            return $data->toArray();
        }

        if (is_array($data)) {
            return $data;
        }

        return null;
    }
}
