<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use Closure;
use Spatie\LaravelData\Data;
use MSML\Blueprint\DSL\Enums\Size;
use MSML\Blueprint\DSL\Enums\Variant;
use MSML\Blueprint\DSL\Nodes\FieldNode;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\Nodes\RepeaterNode;
use MSML\Blueprint\DSL\Contracts\FieldBuilder;
use MSML\Blueprint\DSL\Traits\Common\WithHelp;
use MSML\Blueprint\DSL\Traits\Common\WithSize;
use MSML\Blueprint\DSL\Traits\Common\WithRules;
use MSML\Blueprint\DSL\Traits\Core\NodeUpdater;
use MSML\Blueprint\DSL\Traits\Core\SpanUpdater;
use MSML\Blueprint\DSL\Traits\Common\WithVariant;
use MSML\Blueprint\DSL\Traits\Common\WithDisabled;
use MSML\Blueprint\DSL\Traits\Core\CoreBuilderMethods;
use MSML\Blueprint\DSL\Traits\ValueTypes\HasArrayValue;

final class RepeaterBuilder implements FieldBuilder
{
    use CoreBuilderMethods;
    use HasArrayValue;
    use NodeUpdater;
    use SpanUpdater;
    use WithDisabled;
    use WithHelp;
    use WithRules;
    use WithSize;
    use WithVariant;

    protected RepeaterNode $node;

    /** @var array<string,FieldNode> */
    private array $fieldDefinitions = [];

    /** @var array<string,list<string>> */
    private array $fieldRules = [];

    public function __construct(
        BlueprintCompiler $root,
        int $sectionIndex,
        int $rowIndex,
        string $key,
        string|null $label,
        FieldsetBuilder|PanelBuilder|RowBuilder|TabBuilder $context
    ) {
        $this->root = $root;
        $this->sectionIndex = $sectionIndex;
        $this->rowIndex = $rowIndex;
        $this->key = $key;
        $this->context = $context;
        $this->node = new RepeaterNode(
            label: $label,
            fields: [],
            summaryFields: [],
            actions: ['edit', 'delete'],
            addButtonText: 'Add new item',
            minItems: null,
            maxItems: null,
            items: null,
            span: 12,
            help: null,
            size: Size::SM,
            variant: Variant::Default,
            disabled: false,
        );

        $this->registerInLayout();
        $this->updateNode();
    }

    public function value(mixed $value): self
    {
        $this->root->defaults[$this->key] = $value;

        return $this;
    }

    public function fields(Closure $fn): self
    {
        $nestedCompiler = new BlueprintCompiler("{$this->key}_fields");

        $nestedCompiler->tab('Fields', function (TabBuilder $t) use ($fn) {
            $fn($t);
        });

        $schema = $nestedCompiler->compile();

        $this->fieldDefinitions = $schema->components;
        $this->fieldRules = $schema->rules;

        $fieldsAsArrays = array_map(fn ($node) => $node->toArray(), $this->fieldDefinitions);

        $this->updateNodeWith(['fields' => $fieldsAsArrays]);

        $this->generateArrayValidationRules();

        return $this;
    }

    /**
     * @param  list<string>  $fields
     */
    public function summary(array $fields): self
    {
        $this->updateNodeWith(['summaryFields' => $fields]);

        return $this;
    }

    /**
     * @param  list<string>  $actions
     */
    public function actions(array $actions): self
    {
        $this->updateNodeWith(['actions' => $actions]);

        return $this;
    }

    public function addButtonText(string $text): self
    {
        $this->updateNodeWith(['addButtonText' => $text]);

        return $this;
    }

    public function minItems(int $min): self
    {
        $this->updateNodeWith(['minItems' => $min]);

        $this->root->rules[$this->key] = array_merge(
            $this->root->rules[$this->key] ?? [],
            ["min:{$min}"]
        );

        return $this;
    }

    public function maxItems(int $max): self
    {
        $this->updateNodeWith(['maxItems' => $max]);

        $this->root->rules[$this->key] = array_merge(
            $this->root->rules[$this->key] ?? [],
            ["max:{$max}"]
        );

        return $this;
    }

    /**
     * @param  list<array<string,mixed>>  $items
     */
    public function items(array $items): self
    {
        $this->root->defaults[$this->key] = $items;
        $this->updateNodeWith(['items' => $items]);

        return $this;
    }

    public function itemsFrom(string $dataKey): self
    {
        $source = $this->normalizeDataSource();

        if ($source === null) {
            return $this;
        }

        $items = $this->valueFromPath($source, $dataKey);

        if (!is_array($items)) {
            return $this;
        }

        $this->root->defaults[$this->key] = $items;
        $this->updateNodeWith(['items' => $items]);

        return $this;
    }

    private function generateArrayValidationRules(): void
    {
        $existingRules = $this->root->rules[$this->key] ?? [];

        if (!in_array('array', $existingRules, true)) {
            $this->root->rules[$this->key] = array_merge(
                $existingRules,
                ['array']
            );
        }

        foreach ($this->fieldRules as $fieldKey => $rules) {
            $arrayRuleKey = "{$this->key}.*.{$fieldKey}";
            $this->root->rules[$arrayRuleKey] = $rules;
        }
    }

    /**
     * @param  array<mixed>  $source
     */
    private function valueFromPath(array $source, string $path): mixed
    {
        if ($path === '') {
            return null;
        }

        $segments = explode('.', $path);
        $value = $source;

        foreach ($segments as $segment) {
            if (!is_array($value) || !array_key_exists($segment, $value)) {
                return null;
            }

            $value = $value[$segment];
        }

        return $value;
    }

    /**
     * @return array<mixed>|null
     */
    private function normalizeDataSource(): array|null
    {
        $data = $this->root->data;

        if ($data instanceof Data) {
            return $data->toArray();
        }

        if (is_array($data)) {
            return $data;
        }

        return null;
    }
}
