<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Traits\Specialized;

use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Carbon\CarbonInterval;
use InvalidArgumentException;

trait WithTimeGridProps
{
    public function withSeconds(bool $withSeconds = true): self
    {
        $this->updateNodeWith(['withSeconds' => $withSeconds]);

        return $this;
    }

    public function allowDeselect(bool $allowDeselect = true): self
    {
        $this->updateNodeWith(['allowDeselect' => $allowDeselect]);

        return $this;
    }

    /**
     * @param  array<string>  $data
     */
    public function data(array $data): self
    {
        $this->updateNodeWith(['data' => $data]);

        return $this;
    }

    /**
     * @param  array<string>  $disableTime
     */
    public function disableTime(array $disableTime): self
    {
        $this->updateNodeWith(['disableTime' => $disableTime]);

        return $this;
    }

    public function fromRange(string $start, string $end, string $interval): self
    {
        $startAt = Carbon::createFromFormat('H:i', $start);
        $endAt = Carbon::createFromFormat('H:i', $end);

        if ($startAt === null || $endAt === null) {
            throw new InvalidArgumentException('Start and end must be in HH:MM format.');
        }

        if ($endAt->lt($startAt)) {
            $endAt = $endAt->copy()->addDay();
        }

        $step = $this->intervalFromHHMM($interval);

        $period = CarbonPeriod::create($startAt, $step, $endAt, CarbonPeriod::INCLUDE_END_DATE);

        $slots = [];

        /** @var \Carbon\CarbonInterface $t */
        foreach ($period as $t) {
            $slots[] = $t->format('H:i');
        }

        $this->updateNodeWith(['data' => $slots]);

        return $this;
    }

    private function intervalFromHHMM(string $hhmm): CarbonInterval
    {
        if (!preg_match('/^\d{2}:\d{2}$/', $hhmm)) {
            throw new InvalidArgumentException('Interval must be in HH:MM format.');
        }

        [$h, $m] = array_map('intval', explode(':', $hhmm));
        if ($h === 0 && $m === 0) {
            throw new InvalidArgumentException('Interval cannot be 00:00.');
        }

        if ($h > 23) {
            throw new InvalidArgumentException('Interval hour cannot be higher than 23');
        }

        if ($m > 59) {
            throw new InvalidArgumentException('Interval minutes cannot be higher than 59');
        }

        return CarbonInterval::hours($h)->minutes($m);
    }
}
