<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use MSML\Blueprint\DSL\Enums\Size;
use MSML\Blueprint\DSL\Enums\Variant;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\Nodes\TimePickerNode;
use MSML\Blueprint\DSL\Contracts\FieldBuilder;
use MSML\Blueprint\DSL\Traits\Common\WithHelp;
use MSML\Blueprint\DSL\Traits\Common\WithSize;
use MSML\Blueprint\DSL\Traits\Common\WithRules;
use MSML\Blueprint\DSL\Traits\Core\NodeUpdater;
use MSML\Blueprint\DSL\Traits\Core\SpanUpdater;
use MSML\Blueprint\DSL\Traits\Common\WithVariant;
use MSML\Blueprint\DSL\Traits\Common\WithDisabled;
use MSML\Blueprint\DSL\Traits\Core\CoreBuilderMethods;
use MSML\Blueprint\DSL\Traits\ValueTypes\HasStringValue;
use MSML\Blueprint\DSL\Traits\Specialized\WithTimeBounds;
use MSML\Blueprint\DSL\Traits\Specialized\WithTimePickerProps;

final class TimePickerBuilder implements FieldBuilder
{
    use CoreBuilderMethods;
    use HasStringValue;
    use NodeUpdater;
    use SpanUpdater;
    use WithDisabled;
    use WithHelp;
    use WithRules;
    use WithSize;
    use WithTimeBounds;
    use WithTimePickerProps;
    use WithVariant;

    protected TimePickerNode $node;

    public function __construct(
        BlueprintCompiler $root,
        int $sectionIndex,
        int $rowIndex,
        string $key,
        string|null $label,
        FieldsetBuilder|PanelBuilder|RowBuilder|TabBuilder $context
    ) {
        $this->root = $root;
        $this->sectionIndex = $sectionIndex;
        $this->rowIndex = $rowIndex;
        $this->key = $key;
        $this->context = $context;

        $this->node = new TimePickerNode(
            label: $label,
            span: 12,
            format: '24h',
            minTime: null,
            maxTime: null,
            interval: 15,
            help: null,
            size: Size::SM,
            variant: Variant::Default,
            disabled: false,
            withSeconds: false,
            withDropdown: false,
            hoursStep: 1,
            minutesStep: 1,
            secondsStep: 1,
            maxDropdownContentHeight: 200,
            reverseTimeControlsList: false,
        );

        $this->registerInLayout();
        $this->updateNode();
    }

    public function interval(int $minutes): self
    {
        $this->updateNodeWith(['interval' => $minutes]);

        return $this;
    }

    public function withDropdown(bool $withDropdown = true): self
    {
        $this->updateNodeWith(['withDropdown' => $withDropdown]);

        return $this;
    }

    public function hoursStep(int $step): self
    {
        $this->updateNodeWith(['hoursStep' => $step]);

        return $this;
    }

    public function minutesStep(int $step): self
    {
        $this->updateNodeWith(['minutesStep' => $step]);

        return $this;
    }

    public function secondsStep(int $step): self
    {
        $this->updateNodeWith(['secondsStep' => $step]);

        return $this;
    }

    public function maxDropdownContentHeight(int $height): self
    {
        $this->updateNodeWith(['maxDropdownContentHeight' => $height]);

        return $this;
    }

    public function reverseTimeControlsList(bool $reverse = true): self
    {
        $this->updateNodeWith(['reverseTimeControlsList' => $reverse]);

        return $this;
    }
}
