<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use MSML\Blueprint\DSL\Enums\Size;
use MSML\Blueprint\DSL\Nodes\DividerNode;
use MSML\Blueprint\DSL\Contracts\ElementBuilder;
use MSML\Blueprint\DSL\Schema\DividerElementSchema;

final class DividerBuilder implements ElementBuilder
{
    private int $span = 12;

    private string|null $color = null;

    private string|null $label = null;

    private string $labelPosition = 'center';

    private string $orientation = 'horizontal';

    private Size $size = Size::XS;

    public function color(string|null $color): self
    {
        $this->color = $color;

        return $this;
    }

    public function label(string|null $label): self
    {
        $this->label = $label;

        return $this;
    }

    public function labelPosition(string $position): self
    {
        $this->labelPosition = $position;

        return $this;
    }

    public function orientation(string $orientation): self
    {
        $this->orientation = $orientation;

        return $this;
    }

    public function size(Size|string $size): self
    {
        $this->size = $size instanceof Size ? $size : Size::from($size);

        return $this;
    }

    public function span(int $columns): self
    {
        $this->span = max(1, min(12, $columns));

        return $this;
    }

    public function toDividerElementSchema(): DividerElementSchema
    {
        $node = new DividerNode(
            span: $this->span,
            color: $this->color,
            label: $this->label,
            labelPosition: $this->labelPosition,
            orientation: $this->orientation,
            size: $this->size,
        );

        return new DividerElementSchema($node);
    }
}
