<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use Closure;
use MSML\Blueprint\DSL\Enums\Variant;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\FieldBuilderMethods;
use MSML\Blueprint\DSL\Contracts\ElementBuilder;
use MSML\Blueprint\DSL\Schema\FieldsetElementSchema;
use MSML\Blueprint\DSL\Schema\ComponentElementSchema;

final class FieldsetBuilder implements ElementBuilder
{
    use FieldBuilderMethods;

    private int $span = 12;

    private string|null $legend = null;

    private Variant $variant = Variant::Default;

    /** @var array<RowBuilder|DividerBuilder|AccordionBuilder|ComponentElementSchema> */
    private array $elements = [];

    public function __construct(
        private BlueprintCompiler $root,
        private int $sectionIndex,
        private int $rowIndex,
        string|null $legend = null,
    ) {
        $this->legend = $legend;
    }

    public function legend(string $legend): self
    {
        $this->legend = $legend;

        return $this;
    }

    public function variant(Variant $variant): self
    {
        $this->variant = $variant;

        return $this;
    }

    /**
     * @param  string|Closure(RowBuilder): void|null  $titleOrFn
     * @param  Closure(RowBuilder): void|null  $fn
     */
    public function row(Closure|string|null $titleOrFn = null, Closure|null $fn = null): RowBuilder
    {
        if ($titleOrFn instanceof Closure) {
            $title = null;
            $fn = $titleOrFn;
        } else {
            $title = $titleOrFn;
        }

        $row = new RowBuilder($this->root, $this->sectionIndex, -1, $title);

        if ($fn !== null) {
            $fn($row);
        }

        $this->elements[] = $row;

        return $row;
    }

    public function divider(): DividerBuilder
    {
        $divider = new DividerBuilder;
        $this->elements[] = $divider;

        return $divider;
    }

    /**
     * @param  Closure(AccordionBuilder): void  $fn
     */
    public function accordion(Closure $fn): AccordionBuilder
    {
        $accordion = new AccordionBuilder;
        $fn($accordion);
        $this->elements[] = $accordion;

        return $accordion;
    }

    public function span(int $columns): self
    {
        $this->span = max(1, min(12, $columns));

        return $this;
    }

    protected function getCompiler(): BlueprintCompiler
    {
        return $this->root;
    }

    protected function getSectionIndex(): int
    {
        return $this->sectionIndex;
    }

    protected function getRowIndex(): int
    {
        return $this->rowIndex;
    }

    public function registerComponent(string $key, int $span = 12): void
    {
        $this->elements[] = new ComponentElementSchema($key, $span);
    }

    public function updateComponentSpan(string $key, int $span): void
    {
        foreach ($this->elements as $index => $element) {
            if ($element instanceof ComponentElementSchema && $element->key === $key) {
                $this->elements[$index] = new ComponentElementSchema($key, $span);
                break;
            }
        }
    }

    public function toFieldsetElementSchema(): FieldsetElementSchema
    {
        $elementSchemas = [];

        foreach ($this->elements as $element) {
            if ($element instanceof RowBuilder) {
                $elementSchemas[] = $element->toRowElementSchema();
            } elseif ($element instanceof DividerBuilder) {
                $elementSchemas[] = $element->toDividerElementSchema();
            } elseif ($element instanceof AccordionBuilder) {
                $elementSchemas[] = $element->toAccordionElementSchema();
            } elseif ($element instanceof ComponentElementSchema) {
                $elementSchemas[] = $element;
            }
        }

        return new FieldsetElementSchema(
            elements: $elementSchemas,
            span: $this->span,
            legend: $this->legend,
            variant: $this->variant,
        );
    }

    public function getLegend(): string|null
    {
        return $this->legend;
    }
}
