<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use Closure;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\FieldBuilderMethods;
use MSML\Blueprint\DSL\Contracts\ElementBuilder;
use MSML\Blueprint\DSL\Schema\PanelElementSchema;
use MSML\Blueprint\DSL\Schema\ComponentElementSchema;

final class PanelBuilder implements ElementBuilder
{
    use FieldBuilderMethods;

    private int $span = 12;

    private string|null $title = null;

    private string|null $description = null;

    private bool $collapsible = false;

    private bool $defaultExpanded = true;

    private bool $bordered = true;

    /** @var array<RowBuilder|FieldsetBuilder|DividerBuilder|AccordionBuilder|ComponentElementSchema> */
    private array $elements = [];

    public function __construct(
        private BlueprintCompiler $root,
        private int $sectionIndex,
        private int $rowIndex,
        string|null $title = null,
    ) {
        $this->title = $title;
    }

    public function title(string $title): self
    {
        $this->title = $title;

        return $this;
    }

    public function description(string $description): self
    {
        $this->description = $description;

        return $this;
    }

    public function collapsible(bool $defaultExpanded = true): self
    {
        $this->collapsible = true;
        $this->defaultExpanded = $defaultExpanded;

        return $this;
    }

    public function bordered(bool $bordered = true): self
    {
        $this->bordered = $bordered;

        return $this;
    }

    /**
     * @param  string|Closure(RowBuilder): void|null  $titleOrFn
     * @param  Closure(RowBuilder): void|null  $fn
     */
    public function row(Closure|string|null $titleOrFn = null, Closure|null $fn = null): RowBuilder
    {
        if ($titleOrFn instanceof Closure) {
            $title = null;
            $fn = $titleOrFn;
        } else {
            $title = $titleOrFn;
        }

        $row = new RowBuilder($this->root, $this->sectionIndex, -1, $title);

        if ($fn !== null) {
            $fn($row);
        }

        $this->elements[] = $row;

        return $row;
    }

    /**
     * @param  string|Closure(FieldsetBuilder): void|null  $legendOrFn
     * @param  Closure(FieldsetBuilder): void|null  $fn
     */
    public function fieldset(Closure|string|null $legendOrFn = null, Closure|null $fn = null): FieldsetBuilder
    {
        if ($legendOrFn instanceof Closure) {
            $legend = null;
            $fn = $legendOrFn;
        } else {
            $legend = $legendOrFn;
        }

        $fieldset = new FieldsetBuilder($this->root, $this->sectionIndex, -1, $legend);

        if ($fn !== null) {
            $fn($fieldset);
        }

        $this->elements[] = $fieldset;

        return $fieldset;
    }

    public function divider(): DividerBuilder
    {
        $divider = new DividerBuilder;
        $this->elements[] = $divider;

        return $divider;
    }

    /**
     * @param  Closure(AccordionBuilder): void  $fn
     */
    public function accordion(Closure $fn): AccordionBuilder
    {
        $accordion = new AccordionBuilder;
        $fn($accordion);
        $this->elements[] = $accordion;

        return $accordion;
    }

    public function span(int $columns): self
    {
        $this->span = max(1, min(12, $columns));

        return $this;
    }

    protected function getCompiler(): BlueprintCompiler
    {
        return $this->root;
    }

    protected function getSectionIndex(): int
    {
        return $this->sectionIndex;
    }

    protected function getRowIndex(): int
    {
        return $this->rowIndex;
    }

    public function registerComponent(string $key, int $span = 12): void
    {
        $this->elements[] = new ComponentElementSchema($key, $span);
    }

    public function updateComponentSpan(string $key, int $span): void
    {
        foreach ($this->elements as $index => $element) {
            if ($element instanceof ComponentElementSchema && $element->key === $key) {
                $this->elements[$index] = new ComponentElementSchema($key, $span);
                break;
            }
        }
    }

    public function toPanelElementSchema(): PanelElementSchema
    {
        $elementSchemas = [];

        foreach ($this->elements as $element) {
            if ($element instanceof RowBuilder) {
                $elementSchemas[] = $element->toRowElementSchema();
            } elseif ($element instanceof FieldsetBuilder) {
                $elementSchemas[] = $element->toFieldsetElementSchema();
            } elseif ($element instanceof DividerBuilder) {
                $elementSchemas[] = $element->toDividerElementSchema();
            } elseif ($element instanceof AccordionBuilder) {
                $elementSchemas[] = $element->toAccordionElementSchema();
            } elseif ($element instanceof ComponentElementSchema) {
                $elementSchemas[] = $element;
            }
        }

        return new PanelElementSchema(
            elements: $elementSchemas,
            span: $this->span,
            title: $this->title,
            description: $this->description,
            collapsible: $this->collapsible,
            defaultExpanded: $this->defaultExpanded,
            bordered: $this->bordered
        );
    }

    public function getTitle(): string|null
    {
        return $this->title;
    }
}
