<?php

namespace MSML\Blueprint\DSL;

use Closure;
use Spatie\LaravelData\Data;
use MSML\Blueprint\DSL\Nodes\FieldNode;
use MSML\Blueprint\DSL\Schema\TabSchema;
use MSML\Blueprint\DSL\Builders\TabBuilder;
use MSML\Blueprint\DSL\Schema\BlueprintSchema;

final class BlueprintCompiler
{
    public string $key;

    public int $version = 1;

    /** @var array<string,mixed> */
    public array $defaults = [];

    /** @var array<string,FieldNode> */
    public array $components = [];

    /** @var array<string,list<string>> */
    public array $rules = [];

    /** @var array<TabBuilder> */
    private array $tabs = [];

    /** @var Data|array<string,mixed>|null */
    public mixed $data = null;

    /**
     * @param  Data|array<string,mixed>|null  $data
     */
    public function __construct(string $key, array|Data|null $data = null)
    {
        $this->key = $key;
        $this->ingestData($data);
    }

    /**
     * @param  Data|array<string,mixed>|null  $data
     */
    public static function make(string $key, array|Data|null $data = null): self
    {
        return new self($key, $data);
    }

    public function version(int $v): self
    {
        $this->version = $v;

        return $this;
    }

    /**
     * @param  array<string,mixed>  $defaults
     */
    public function defaults(array $defaults): self
    {
        $this->defaults = array_replace($this->defaults, $defaults);

        return $this;
    }

    /**
     * @param  Closure(TabBuilder): void|null  $fn
     */
    public function tab(string $title, Closure|null $fn = null): self
    {
        $tabBuilder = new TabBuilder($this, count($this->tabs));
        $this->tabs[$title] = $tabBuilder;

        if ($fn !== null) {
            $fn($tabBuilder);
        }

        return $this;
    }

    public function compile(): BlueprintSchema
    {
        $tabSchemas = [];

        foreach ($this->tabs as $title => $tabBuilder) {
            $tabSchemas[] = new TabSchema($title, $tabBuilder->toRowSchemas(), $tabBuilder->getBadge());
        }

        return new BlueprintSchema(
            key: $this->key,
            version: $this->version,
            defaults: $this->defaults,
            components: $this->components,
            rules: $this->rules,
            layout: $tabSchemas,
        );
    }

    /**
     * @param  Data|array<string,mixed>|null  $data
     */
    private function ingestData(array|Data|null $data): void
    {
        $this->data = $data;

        if ($data instanceof Data) {
            /** @var array<string, mixed> $dataArray */
            $dataArray = $data->toArray();
            $this->defaults = array_replace($this->defaults, $dataArray);
        } elseif (is_array($data)) {
            $this->defaults = array_replace($this->defaults, $data);
        }
    }
}
