<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use Spatie\LaravelData\Data;
use MSML\Blueprint\DSL\Enums\Size;
use MSML\Blueprint\DSL\Enums\Variant;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\Contracts\FieldBuilder;
use MSML\Blueprint\DSL\Nodes\AutoCompleteNode;
use MSML\Blueprint\DSL\Traits\Common\WithHelp;
use MSML\Blueprint\DSL\Traits\Common\WithSize;
use MSML\Blueprint\DSL\Traits\Common\WithRules;
use MSML\Blueprint\DSL\Traits\Core\NodeUpdater;
use MSML\Blueprint\DSL\Traits\Core\SpanUpdater;
use MSML\Blueprint\DSL\Traits\Common\WithVariant;
use MSML\Blueprint\DSL\Traits\Common\WithDisabled;
use MSML\Blueprint\DSL\Traits\Common\WithMultiple;
use MSML\Blueprint\DSL\Traits\Common\WithPlaceholder;
use MSML\Blueprint\DSL\Traits\Core\CoreBuilderMethods;
use MSML\Blueprint\DSL\Traits\Specialized\WithRelationEndpoint;

final class RelationBuilder implements FieldBuilder
{
    use CoreBuilderMethods;
    use NodeUpdater;
    use SpanUpdater;
    use WithDisabled;
    use WithHelp;
    use WithMultiple;
    use WithPlaceholder;
    use WithRelationEndpoint;
    use WithRules;
    use WithSize;
    use WithVariant;

    protected AutoCompleteNode $node;

    public function __construct(
        BlueprintCompiler $root,
        int $sectionIndex,
        int $rowIndex,
        string $key,
        string|null $label,
        FieldsetBuilder|PanelBuilder|RowBuilder|TabBuilder $context
    ) {
        $this->root = $root;
        $this->sectionIndex = $sectionIndex;
        $this->rowIndex = $rowIndex;
        $this->key = $key;
        $this->context = $context;
        $this->node = new AutoCompleteNode(
            label: $label,
            endpoint: null,
            selected: null,
            span: 12,
            scope: null,
            multiple: false,
            debounceMs: 250,
            placeholder: null,
            help: null,
            size: Size::SM,
            variant: Variant::Default,
            disabled: false,
        );

        $this->registerInLayout();
        $this->updateNode();
    }

    public function value(mixed $value): self
    {
        $this->root->defaults[$this->key] = $value;

        return $this;
    }

    public function selectedOption(int|string $value, string $label): self
    {
        $this->root->defaults[$this->key] = $value;

        $this->updateNodeWith([
            'selected' => [
                'value' => $value,
                'label' => $label,
            ],
        ]);

        return $this;
    }

    public function selectedOptionFrom(string $dataKey, string $labelKey = 'label', string $valueKey = 'value'): self
    {
        $source = $this->normalizeDataSource();

        if ($source === null) {
            return $this;
        }

        $payload = $this->valueFromPath($source, $dataKey);

        if (!is_array($payload)) {
            return $this;
        }

        $value = $this->valueFromPath($payload, $valueKey);
        $label = $this->valueFromPath($payload, $labelKey);

        if (!is_string($label) && !is_numeric($label)) {
            return $this;
        }

        $labelString = is_string($label) ? $label : (string)$label;

        $this->root->defaults[$this->key] = $value;

        $this->updateNodeWith([
            'selected' => [
                'value' => $value,
                'label' => $labelString,
            ],
        ]);

        return $this;
    }

    /**
     * @param  array<mixed>  $source
     */
    private function valueFromPath(array $source, string $path): mixed
    {
        if ($path === '') {
            return null;
        }

        $segments = explode('.', $path);
        $value = $source;

        foreach ($segments as $segment) {
            if (!is_array($value) || !array_key_exists($segment, $value)) {
                return null;
            }

            $value = $value[$segment];
        }

        return $value;
    }

    /**
     * @return array<mixed>|null
     */
    private function normalizeDataSource(): array|null
    {
        $data = $this->root->data;

        if ($data instanceof Data) {
            return $data->toArray();
        }

        if (is_array($data)) {
            return $data;
        }

        return null;
    }
}
