<?php

namespace MSML\Blueprint\DSL\Builders;

use Closure;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\FieldBuilderMethods;
use MSML\Blueprint\DSL\Schema\RowElementSchema;

final class TabBuilder
{
    use FieldBuilderMethods;

    /** @var array<RowBuilder|PanelBuilder|FieldsetBuilder|DividerBuilder|AccordionBuilder> */
    private array $elements = [];

    private FieldsetBuilder|PanelBuilder|RowBuilder|null $currentElement = null;

    private int|string|null $badge = null;

    public function __construct(
        private BlueprintCompiler $root,
        private int $sectionIndex
    ) {}

    public function badge(int|string $badge): self
    {
        $this->badge = $badge;

        return $this;
    }

    public function getBadge(): int|string|null
    {
        return $this->badge;
    }

    /**
     * @param  string|Closure(RowBuilder): void|null  $titleOrFn
     * @param  Closure(RowBuilder): void|null  $fn
     */
    public function row(Closure|string|null $titleOrFn = null, Closure|null $fn = null): RowBuilder
    {
        if ($titleOrFn instanceof Closure) {
            $title = null;
            $fn = $titleOrFn;
        } else {
            $title = $titleOrFn;
        }

        $row = new RowBuilder($this->root, $this->sectionIndex, count($this->elements), $title);
        $this->elements[] = $row;
        $this->currentElement = $row;

        if ($fn !== null) {
            $fn($row);
            $this->currentElement = null;
        }

        return $row;
    }

    /**
     * @param  string|Closure(PanelBuilder): void|null  $titleOrFn
     * @param  Closure(PanelBuilder): void|null  $fn
     */
    public function panel(Closure|string|null $titleOrFn = null, Closure|null $fn = null): PanelBuilder
    {
        if ($titleOrFn instanceof Closure) {
            $title = null;
            $fn = $titleOrFn;
        } else {
            $title = $titleOrFn;
        }

        $panel = new PanelBuilder($this->root, $this->sectionIndex, count($this->elements), $title);
        $this->elements[] = $panel;
        $this->currentElement = $panel;

        if ($fn !== null) {
            $fn($panel);
            $this->currentElement = null;
        }

        return $panel;
    }

    /**
     * @param  string|Closure(FieldsetBuilder): void|null  $legendOrFn
     * @param  Closure(FieldsetBuilder): void|null  $fn
     */
    public function fieldset(Closure|string|null $legendOrFn = null, Closure|null $fn = null): FieldsetBuilder
    {
        if ($legendOrFn instanceof Closure) {
            $legend = null;
            $fn = $legendOrFn;
        } else {
            $legend = $legendOrFn;
        }

        $fieldset = new FieldsetBuilder($this->root, $this->sectionIndex, count($this->elements), $legend);
        $this->elements[] = $fieldset;
        $this->currentElement = $fieldset;

        if ($fn !== null) {
            $fn($fieldset);
            $this->currentElement = null;
        }

        return $fieldset;
    }

    public function divider(): DividerBuilder
    {
        $divider = new DividerBuilder;
        $this->elements[] = $divider;
        $this->currentElement = null;

        return $divider;
    }

    /**
     * @param  Closure(AccordionBuilder): void  $fn
     */
    public function accordion(Closure $fn): AccordionBuilder
    {
        $accordion = new AccordionBuilder;
        $fn($accordion);
        $this->elements[] = $accordion;
        $this->currentElement = null;

        return $accordion;
    }

    /**
     * Convert all elements to schemas
     *
     * @return list<RowElementSchema>
     */
    public function toRowSchemas(): array
    {
        $schemas = [];

        foreach ($this->elements as $element) {
            if ($element instanceof RowBuilder) {
                $schemas[] = $element->toRowElementSchema();
            } elseif ($element instanceof PanelBuilder) {
                $panelSchema = $element->toPanelElementSchema();
                $schemas[] = new RowElementSchema([$panelSchema], 12);
            } elseif ($element instanceof FieldsetBuilder) {
                $fieldsetSchema = $element->toFieldsetElementSchema();
                $schemas[] = new RowElementSchema([$fieldsetSchema], 12);
            } elseif ($element instanceof DividerBuilder) {
                $dividerSchema = $element->toDividerElementSchema();
                $schemas[] = new RowElementSchema([$dividerSchema], 12);
            } elseif ($element instanceof AccordionBuilder) {
                $accordionSchema = $element->toAccordionElementSchema();
                $schemas[] = new RowElementSchema([$accordionSchema], 12);
            }
        }

        return $schemas;
    }

    protected function getCompiler(): BlueprintCompiler
    {
        return $this->root;
    }

    protected function getSectionIndex(): int
    {
        return $this->sectionIndex;
    }

    protected function getRowIndex(): int
    {
        if ($this->currentElement === null) {
            $this->row();
        }

        return count($this->elements) - 1;
    }

    public function getCurrentRow(): RowBuilder
    {
        if ($this->currentElement === null || !($this->currentElement instanceof RowBuilder)) {
            $this->row();
        }

        /** @var RowBuilder $currentElement */
        $currentElement = $this->currentElement;

        return $currentElement;
    }
}
