<?php

declare(strict_types=1);

namespace MSML\Blueprint\DSL\Builders;

use Spatie\LaravelData\Data;
use MSML\Blueprint\DSL\Enums\Size;
use MSML\Blueprint\DSL\Enums\Variant;
use MSML\Blueprint\DSL\Nodes\TagsNode;
use MSML\Blueprint\DSL\BlueprintCompiler;
use MSML\Blueprint\DSL\Contracts\FieldBuilder;
use MSML\Blueprint\DSL\Traits\Common\WithHelp;
use MSML\Blueprint\DSL\Traits\Common\WithSize;
use MSML\Blueprint\DSL\Traits\Common\WithRules;
use MSML\Blueprint\DSL\Traits\Core\NodeUpdater;
use MSML\Blueprint\DSL\Traits\Core\SpanUpdater;
use MSML\Blueprint\DSL\Traits\Common\WithVariant;
use MSML\Blueprint\DSL\Traits\Common\WithDisabled;
use MSML\Blueprint\DSL\Traits\Common\WithPlaceholder;
use MSML\Blueprint\DSL\Traits\Core\CoreBuilderMethods;
use MSML\Blueprint\DSL\Traits\ValueTypes\HasArrayValue;
use MSML\Blueprint\DSL\Traits\Specialized\WithSelectedCount;
use MSML\Blueprint\DSL\Traits\Specialized\WithRelationEndpoint;

final class TagsBuilder implements FieldBuilder
{
    use CoreBuilderMethods;
    use HasArrayValue;
    use NodeUpdater;
    use SpanUpdater;
    use WithDisabled;
    use WithHelp;
    use WithPlaceholder;
    use WithRelationEndpoint;
    use WithRules;
    use WithSelectedCount;
    use WithSize;
    use WithVariant;

    protected TagsNode $node;

    public function __construct(
        BlueprintCompiler $root,
        int $sectionIndex,
        int $rowIndex,
        string $key,
        string|null $label,
        FieldsetBuilder|PanelBuilder|RowBuilder|TabBuilder $context
    ) {
        $this->root = $root;
        $this->sectionIndex = $sectionIndex;
        $this->rowIndex = $rowIndex;
        $this->key = $key;
        $this->context = $context;
        $this->node = new TagsNode(
            label: $label,
            endpoint: null,
            selected: null,
            span: 12,
            maxTags: null,
            suggestions: null,
            debounceMs: 250,
            placeholder: null,
            help: null,
            size: Size::SM,
            variant: Variant::Default,
            disabled: false,
            clearable: true,
            searchable: true,
            showSelectedCount: false,
            selectedCountSingular: null,
            selectedCountPlural: null,
        );

        $this->registerInLayout();
        $this->updateNode();
    }

    public function value(mixed $value): self
    {
        $this->root->defaults[$this->key] = $value;

        return $this;
    }

    /**
     * @param  list<mixed>  $values
     */
    public function selected(array $values): self
    {
        $options = [];

        foreach ($values as $val) {
            $labelString = is_string($val) || is_numeric($val) ? (string)$val : '';
            $options[] = [
                'value' => $val,
                'label' => $labelString,
            ];
        }

        $this->root->defaults[$this->key] = $values;

        $this->updateNodeWith(['selected' => $options]);

        return $this;
    }

    /**
     * @param  list<array{value:mixed,label:string}>  $options
     */
    public function selectedOptions(array $options): self
    {
        $values = [];

        foreach ($options as $opt) {
            $values[] = $opt['value'];
        }

        $this->root->defaults[$this->key] = $values;

        $this->updateNodeWith(['selected' => $options]);

        return $this;
    }

    public function selectedFrom(string $dataKey, string $labelKey = 'label', string $valueKey = 'value'): self
    {
        $source = $this->normalizeDataSource();

        if ($source === null) {
            return $this;
        }

        $items = $this->valueFromPath($source, $dataKey);

        if (!is_array($items)) {
            return $this;
        }

        $options = [];
        $values = [];

        foreach ($items as $item) {
            if (!is_array($item)) {
                continue;
            }

            $value = $this->valueFromPath($item, $valueKey);
            $label = $this->valueFromPath($item, $labelKey);

            if (!is_string($label) && !is_numeric($label)) {
                continue;
            }

            $labelString = is_string($label) ? $label : (string)$label;

            $options[] = [
                'value' => $value,
                'label' => $labelString,
            ];

            $values[] = $value;
        }

        $this->root->defaults[$this->key] = $values;

        $this->updateNodeWith(['selected' => $options]);

        return $this;
    }

    public function maxTags(int $max): self
    {
        $this->updateNodeWith(['maxTags' => $max]);

        return $this;
    }

    /**
     * @param  list<string|array{label:string,value:mixed}>  $suggestions
     */
    public function suggestions(array $suggestions): self
    {
        $this->updateNodeWith(['suggestions' => $suggestions]);

        return $this;
    }

    public function debounce(int $ms): self
    {
        $this->updateNodeWith(['debounceMs' => $ms]);

        return $this;
    }

    public function clearable(bool $clearable = true): self
    {
        $this->updateNodeWith(['clearable' => $clearable]);

        return $this;
    }

    public function searchable(bool $searchable = true): self
    {
        $this->updateNodeWith(['searchable' => $searchable]);

        return $this;
    }

    /**
     * @param  array<mixed>  $source
     */
    private function valueFromPath(array $source, string $path): mixed
    {
        if ($path === '') {
            return null;
        }

        $segments = explode('.', $path);
        $value = $source;

        foreach ($segments as $segment) {
            if (!is_array($value) || !array_key_exists($segment, $value)) {
                return null;
            }

            $value = $value[$segment];
        }

        return $value;
    }

    /**
     * @return array<mixed>|null
     */
    private function normalizeDataSource(): array|null
    {
        $data = $this->root->data;

        if ($data instanceof Data) {
            return $data->toArray();
        }

        if (is_array($data)) {
            return $data;
        }

        return null;
    }
}
