<?php

namespace MSML\Calendar\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Attributes\Scope;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * @property int $id Primary key
 * @property int|null $user_id User who owns this event
 * @property int|null $category_id Category of this event
 * @property string $title Event title
 * @property string|null $description Event description
 * @property Carbon|string $start Start date and time
 * @property Carbon|string $end End date and time
 * @property bool $all_day Whether this is an all-day event
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read \Illuminate\Database\Eloquent\Model|null $user Event owner
 * @property-read \Illuminate\Database\Eloquent\Model|null $category Event category
 *
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent newModelQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent newQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent query()
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereUserId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereCategoryId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereTitle($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereDescription($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereStart($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereEnd($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereAllDay($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereCreatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent whereUpdatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|CalendarEvent range(string|Carbon $start, string|Carbon $end)
 */
class CalendarEvent extends Model
{
    protected $guarded = [];

    protected function casts(): array
    {
        return [
            'start' => 'datetime',
            'end' => 'datetime',
            'all_day' => 'boolean',
        ];
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(config('auth.providers.users.model'));
    }

    public function category(): BelongsTo
    {
        return $this->belongsTo(config('calendar.event_category_model'));
    }

    #[Scope]
    public function range(Builder $query, string|Carbon $start, string|Carbon $end): Builder
    {
        $startDate = $start instanceof Carbon
            ? $start
            : Carbon::parse($start);

        $endDate = $end instanceof Carbon
            ? $end
            : Carbon::parse($end);

        return $query
            ->whereDate('start', '>=', $startDate)
            ->whereDate('end', '<=', $endDate);
    }
}
