<?php

declare(strict_types=1);

namespace MSML\CloudFrontCookies;

use Aws\CloudFront\CookieSigner;
use Illuminate\Auth\Events\Logout;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;
use MSML\CloudFrontCookies\Http\Middleware\EnsureCloudFrontCookies;
use MSML\CloudFrontCookies\Listeners\ClearCloudFrontCookiesOnLogout;
use MSML\CloudFrontCookies\Contracts\CloudFrontCookies as CloudFrontCookiesContract;

class CloudFrontCookiesServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        $package
            ->name('laravel-cloudfront-cookies')
            ->hasConfigFile();
    }

    public function registeringPackage(): void
    {
        $this->app->singleton(CookieSigner::class, function () {
            $keyPairId = config('cloudfront-cookies.key_pair_id');
            $privateKeyPath = config('cloudfront-cookies.private_key_path');

            if (!$keyPairId) {
                throw new \RuntimeException('cloudfront-cookies.key_pair_id is not set.');
            }

            if (!$privateKeyPath) {
                throw new \RuntimeException('cloudfront-cookies.private_key_path is not set.');
            }

            if (!is_file($privateKeyPath)) {
                throw new \RuntimeException("CloudFront private key file not found at: {$privateKeyPath}");
            }

            if (!is_readable($privateKeyPath)) {
                throw new \RuntimeException("CloudFront private key file is not readable at: {$privateKeyPath}");
            }

            return new CookieSigner($keyPairId, $privateKeyPath);
        });

        $this->app->singleton(CloudFrontCookiesContract::class, CloudFrontCookies::class);
    }

    public function bootingPackage(): void
    {
        $this->app['router']->aliasMiddleware('ensure.cloudfront.cookies', EnsureCloudFrontCookies::class);

        $this->app['events']->listen(
            Logout::class,
            ClearCloudFrontCookiesOnLogout::class
        );
    }
}
