<?php

declare(strict_types=1);

namespace MSML\CloudFrontCookies;

use DateTimeInterface;
use Illuminate\Support\Arr;
use Aws\CloudFront\CookieSigner;
use Illuminate\Support\Facades\Cookie;
use MSML\CloudFrontCookies\Contracts\CloudFrontCookies as Contract;

class CloudFrontCookies implements Contract
{
    public const COOKIES = [
        'CloudFront-Policy',
        'CloudFront-Signature',
        'CloudFront-Key-Pair-Id',
    ];

    public function __construct(
        private readonly CookieSigner $signer
    ) {}

    /**
     * @param  string|array<int, string>  $resources  Relative paths (e.g. "foo/*") or absolute URLs.
     * @return array{
     *     'CloudFront-Policy': string,
     *     'CloudFront-Signature': string,
     *     'CloudFront-Key-Pair-Id': string
     * }
     */
    public function make(array|string $resources, DateTimeInterface $expiresAt): array
    {
        $policy = $this->buildPolicy(
            $this->normalizeResources(Arr::wrap($resources)),
            $expiresAt
        );

        return $this->signer->getSignedCookie(null, null, $policy);
    }

    /**
     * @param  array<string, string>  $cookies
     */
    public function queue(array $cookies): void
    {
        $cfg = config('cloudfront-cookies.cookie');

        foreach ($cookies as $name => $value) {
            Cookie::queue(
                name: $name,
                value: $value,
                minutes: (int)($cfg['minutes'] ?? 60),
                path: $cfg['path'] ?? '/',
                domain: $cfg['domain'] ?? null,
                secure: (bool)($cfg['secure'] ?? true),
                httpOnly: (bool)($cfg['http_only'] ?? true),
                raw: false,
                sameSite: $cfg['same_site'] ?? null,
            );
        }
    }

    public function clear(): void
    {
        $cfg = config('cloudfront-cookies.cookie');

        foreach (self::COOKIES as $name) {
            Cookie::queue(
                name: $name,
                value: '',
                minutes: -2628000,
                path: $cfg['path'] ?? '/',
                domain: $cfg['domain'] ?? null,
                secure: (bool)($cfg['secure'] ?? true),
                httpOnly: (bool)($cfg['http_only'] ?? true),
                raw: false,
                sameSite: $cfg['same_site'] ?? null,
            );
        }
    }

    /**
     * @param  array<int, string>  $resources  Fully-qualified resource URLs.
     */
    protected function buildPolicy(array $resources, DateTimeInterface $expiresAt): string
    {
        $statements = array_map(function (string $resource) use ($expiresAt) {
            return [
                'Resource'  => $resource,
                'Condition' => [
                    'DateLessThan' => [
                        'AWS:EpochTime' => $expiresAt->getTimestamp(),
                    ],
                ],
            ];
        }, $resources);

        return json_encode(['Statement' => $statements], JSON_UNESCAPED_SLASHES);
    }

    /**
     * @param  array<int, string>  $resources
     * @return array<int, string>
     *
     * @throws \RuntimeException If no CloudFront domain is configured.
     */
    protected function normalizeResources(array $resources): array
    {
        $scheme = config('cloudfront-cookies.scheme', 'https');
        $domain = config('cloudfront-cookies.domain');

        if (!$domain) {
            throw new \RuntimeException('cloudfront-cookies.domain is not configured.');
        }

        $base = rtrim("{$scheme}://{$domain}", '/');

        return array_map(
            fn (string $resource) => str_contains($resource, '://')
                ? $resource
                : $base . '/' . ltrim($resource, '/'),
            $resources
        );
    }
}
