# Laravel Media Manager

A comprehensive Laravel media management package that provides namespace-based organization, hierarchical folder structures, and a flexible validation rules engine for advanced media asset management.

## Features

- **Namespace-based Organization**: Organize media assets into separate namespaces with isolated configuration
- **Hierarchical Folder Structure**: Create nested folder hierarchies within namespaces
- **Polymorphic Relationships**: Attach media assets to any Eloquent model using pivot tables
- **Flexible Storage**: Support for multiple storage disks (local, S3, etc.) per namespace
- **Rules Engine**: Powerful validation system with configurable rules for uploads and operations
- **Multilingual Support**: Translatable error messages and validation feedback
- **Built on Spatie Media Library**: Leverage the power and reliability of Spatie's proven media handling

## Installation

You can install the package via composer:

```bash
composer require msml/laravel-media-manager
```

You can publish and run the migrations with:

```bash
php artisan vendor:publish --tag="media-manager-migrations"
php artisan migrate
```

You can publish the config file with:

```bash
php artisan vendor:publish --tag="media-manager-config"
```

You can publish the translation files with:

```bash
php artisan vendor:publish --tag="media-manager-translations"
```

This is the contents of the published config file:

```php
return [
    // Configuration options will be documented here
];
```

## Usage

### Adding the Trait to Your Models

First, add the `HasAssets` trait to any model that should manage media assets and define its namespace configuration:

```php
use MSML\MediaManager\Traits\HasAssets;
use MSML\MediaManager\ValueObjects\NamespaceConfiguration;

class Product extends Model
{
    use HasAssets;

    public function getNamespaceConfiguration(): NamespaceConfiguration
    {
        return NamespaceConfiguration::create('products')
            ->maxFileSize(5 * 1024 * 1024) // 5MB limit
            ->allowedMimeTypes(['image/jpeg', 'image/png'])
            ->allowedExtensions(['jpg', 'jpeg', 'png'])
            ->disallowRootFiles() // Files must be in folders
            ->maxFiles(20);
    }
}
```

### Basic Usage

```php
// Upload a file to the model's namespace
$product = Product::find(1);
$asset = $product->attachAsset($uploadedFile);

// Upload to a specific folder
$asset = $product->attachAsset($uploadedFile, $folderId);

// Upload with collection and ordering
$asset = $product->attachAsset($uploadedFile, null, 'gallery', 1);

// Get assets by collection
$galleryAssets = $product->getAssetsByCollection('gallery');

// Move an asset to a different folder
$product->moveAsset($asset, $newFolderId);

// Download an asset
return $product->downloadAsset($asset);
```

### Working with Namespaces

Namespaces are automatically created when referenced by models, but you can also create them manually:

```php
use MSML\MediaManager\Models\MediaNamespace;

// Create a namespace with specific storage disk
$namespace = MediaNamespace::create([
    'name' => 'Product Images',
    'slug' => 'products',
    'disk' => 's3', // Storage disk to use
]);

// The namespace configuration and rules are defined in your model's
// getNamespaceConfiguration() method, not stored in the database
```

### Working with Asset Folders

```php
use MSML\MediaManager\Models\AssetFolder;

// Create a folder structure
$parentFolder = AssetFolder::create([
    'media_namespace_id' => $namespace->id,
    'name' => 'Categories',
]);

$childFolder = AssetFolder::create([
    'media_namespace_id' => $namespace->id,
    'parent_id' => $parentFolder->id,
    'name' => 'Electronics',
]);

// Get folder path
echo $childFolder->path; // "products/Categories/Electronics"

// Check folder relationships
$isAncestor = $parentFolder->isAncestorOf($childFolder); // true
$ancestors = $childFolder->getAncestors(); // Collection of parent folders
```

## Customization

### Translations

The package supports multilingual error messages and notifications. By default, it includes Dutch and English translations. You can customize these messages by publishing the translation files:

```bash
php artisan vendor:publish --tag="media-manager-translations"
```

This will publish translation files to `resources/lang/vendor/media-manager/` in your application. You can then modify or add translations for different languages.

Available translation keys include:
- `rules.allowed_extensions.fail` - Error when file extension is not allowed
- `rules.max_file_size.fail` - Error when file size exceeds limit
- `rules.allowed_mime_types.fail` - Error when MIME type is not allowed
- `rules.allow_root_files.fail` - Error when root files are not allowed
- `rules.image_dimensions.fail` - Error when image dimensions exceed limits
- `rules.max_files.fail` - Error when file count limit is reached

Example translation file (`resources/lang/vendor/media-manager/en/rules.php`):

```php
return [
    'allowed_extensions' => [
        'fail' => 'File extension ":extension" is not allowed. Allowed extensions: :allowed_extensions',
    ],
    'max_file_size' => [
        'fail' => 'File is too large. Maximum: :max_size, Your file: :actual_size',
    ],
    // ... more translations
];
```

### Validation Rules Engine

The package features a powerful rules engine that validates file operations. Rules are configured per model via the `NamespaceConfiguration` and automatically applied during uploads and other operations.

#### Available Rules

- **AllowedExtensionsRule**: Restricts file uploads by extension (`->allowedExtensions(['jpg', 'png'])`)
- **MaxFileSizeRule**: Limits file size in bytes (`->maxFileSize(5 * 1024 * 1024)`)
- **AllowedMimeTypesRule**: Restricts files by MIME type (`->allowedMimeTypes(['image/jpeg'])`)
- **AllowRootFilesRule**: Controls root uploads (`->disallowRootFiles()`)
- **ImageDimensionsRule**: Limits image dimensions (`->maxImageDimensions(1920, 1080)`)
- **MaxFilesRule**: Limits total files per namespace (`->maxFiles(100)`)

#### Custom Rules

You can create custom rules by implementing the `Rule` interface:

```php
use MSML\MediaManager\Rules\Rule;
use MSML\MediaManager\Rules\RuleContext;
use MSML\MediaManager\Rules\RuleResult;

class CustomValidationRule implements Rule
{
    public function evaluate(RuleContext $context): RuleResult
    {
        // Your validation logic here
        if ($this->isValid($context)) {
            return RuleResult::pass();
        }
        
        return RuleResult::fail(__('custom-package::validation.failed'));
    }
}

// Add to your namespace configuration
public function getNamespaceConfiguration(): NamespaceConfiguration
{
    return NamespaceConfiguration::create('products')
        ->addRule(new CustomValidationRule());
}
```

#### Rule Context

Rules receive a `RuleContext` object containing:
- File information (size, MIME type, filename)
- Upload context (namespace, folder, target model)
- Action being performed (upload, move, etc.)
- Current file counts and metadata

#### Error Handling

The rules engine aggregates all validation errors and provides detailed feedback:

```php
$result = $product->getNamespaceConfiguration()->validateAction($context);

if ($result->failed()) {
    $errors = $result->getErrors(); // Array of error messages
    $firstError = $result->getFirstError(); // String or null
}
```

## Testing

```bash
composer test
```

## Changelog

Please see [CHANGELOG](CHANGELOG.md) for more information on what has changed recently.

## Contributing

Please see [CONTRIBUTING](CONTRIBUTING.md) for details.

## Security Vulnerabilities

Please review [our security policy](../../security/policy) on how to report security vulnerabilities.

## Credits

- [Dylan-MSML](https://github.com/MSMLBV)
- [All Contributors](../../contributors)

## License

The MIT License (MIT). Please see [License File](LICENSE.md) for more information.
