<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use InvalidArgumentException;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;
use MSML\MediaManager\Models\MediaNamespace;

final class UploadFileAction
{
    public function __construct(
        private readonly ResolveFolderFromPathAction $resolveFolderFromPathAction
    ) {}

    /**
     * @param  array{width: int, height: int, fit: string, quality: int, format: string|null}  $thumbnailConversionSettings
     *
     * @throws InvalidArgumentException
     */
    public function handle(
        string $namespaceSlug,
        UploadedFile $file,
        string|null $folderPath = null,
        array $thumbnailConversionSettings = ['width' => 300, 'height' => 300, 'fit' => 'crop', 'quality' => 80, 'format' => null]
    ): Asset {
        return DB::transaction(function () use ($namespaceSlug, $file, $folderPath, $thumbnailConversionSettings) {
            $namespace = MediaNamespace::whereSlug($namespaceSlug)->firstOrFail();

            $folder = null;

            if (!$namespace->disk) {
                throw new InvalidArgumentException('No disk configured');
            }

            $folder = $this->resolveFolderFromPathAction->handle($namespace, $folderPath);

            $metadata = [
                'original_name' => $file->getClientOriginalName(),
                'uploaded_by'   => auth()->id(),
            ];

            if (str_starts_with($file->getMimeType() ?? '', 'image/')) {
                $imageSize = getimagesize($file->getPathname());
                if ($imageSize !== false) {
                    $metadata['width'] = $imageSize[0];
                    $metadata['height'] = $imageSize[1];
                }

                $metadata['conversions'] = [
                    'thumb' => $thumbnailConversionSettings,
                ];
            }

            /** @var Asset */
            $asset = Asset::create([
                'media_namespace_id' => $namespace->id,
                'asset_folder_id'    => $folder?->id,
                'metadata'           => $metadata,
            ]);

            $asset->addMedia($file)
                ->toMediaCollection($namespace->name, $namespace->disk);

            return $asset;
        });
    }
}
