<?php

declare(strict_types=1);

namespace MSML\MediaManager\Models;

use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;

/**
 * @property int $id
 * @property int $media_namespace_id
 * @property int|null $parent_id
 * @property string $name
 * @property Carbon $created_at
 * @property Carbon $updated_at
 * @property-read string $path
 * @property-read string $full_path
 * @property-read bool $is_root
 * @property-read int $depth
 * @property-read MediaNamespace $namespace
 * @property-read AssetFolder|null $parent
 * @property-read EloquentCollection<int, AssetFolder> $children
 * @property-read EloquentCollection<int, Asset> $assets
 *
 * @mixin \Eloquent
 */
class AssetFolder extends Model
{
    /**
     * @var list<string>
     */
    protected $fillable = [
        'media_namespace_id',
        'parent_id',
        'name',
    ];

    protected $table = 'asset_folders';

    /**
     * @var array<string, string>
     */
    protected $casts = [
        'media_namespace_id' => 'integer',
        'parent_id'          => 'integer',
        'created_at'         => 'datetime',
        'updated_at'         => 'datetime',
    ];

    /**
     * @var list<string>
     */
    protected $appends = [
        'path',
        'full_path',
    ];

    /**
     * @return BelongsTo<MediaNamespace, $this>
     */
    public function namespace(): BelongsTo
    {
        return $this->belongsTo(MediaNamespace::class, 'media_namespace_id');
    }

    /**
     * @return BelongsTo<self, $this>
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    /**
     * @return HasMany<self, $this>
     */
    public function children(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    /**
     * @return HasMany<Asset, $this>
     */
    public function assets(): HasMany
    {
        return $this->hasMany(Asset::class, 'asset_folder_id');
    }

    /**
     * @return list<string>
     */
    public function getPathSegments(): array
    {
        $segments = [];
        $folder = $this;

        while ($folder) {
            array_unshift($segments, $folder->name);
            $folder = $folder->parent;
        }

        return $segments;
    }

    /**
     * @return Collection<int, AssetFolder>
     */
    public function getAncestors(): Collection
    {
        /** @var Collection<int, AssetFolder> $ancestors */
        $ancestors = collect();
        $parent = $this->parent;

        while ($parent) {
            $ancestors->push($parent);
            $parent = $parent->parent;
        }

        return $ancestors->reverse();
    }

    /**
     * @return Collection<int, AssetFolder>
     */
    public function getAllDescendants(): Collection
    {
        return $this->children->flatMap(
            fn (self $child): Collection => $child->getAllDescendants()->prepend($child)
        );
    }

    public function isAncestorOf(self $folder): bool
    {
        return $folder->getAncestors()->contains('id', $this->id);
    }

    public function isDescendantOf(self $folder): bool
    {
        return $this->getAncestors()->contains('id', $folder->id);
    }

    /**
     * @return Attribute<non-falsy-string, never>
     */
    protected function path(): Attribute
    {
        return Attribute::get(function () {
            $segments = $this->getPathSegments();

            return $this->namespace->slug . '/' . implode('/', $segments);
        });
    }

    /**
     * @return Attribute<non-falsy-string, void>
     */
    protected function fullPath(): Attribute
    {
        return Attribute::get(
            fn (): string => $this->path
        );
    }

    /**
     * @return Attribute<bool, void>
     */
    protected function isRoot(): Attribute
    {
        return Attribute::get(fn (): bool => is_null($this->parent_id));
    }

    /**
     * @return Attribute<int<-1, max>, void>
     */
    protected function depth(): Attribute
    {
        return Attribute::get(function (): int {
            return count($this->getPathSegments()) - 1;
        });
    }
}
