<?php

declare(strict_types=1);

namespace MSML\MediaManager\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * @property int $id
 * @property string $name
 * @property string $slug
 * @property string|null $disk
 * @property \Illuminate\Support\Carbon $created_at
 * @property \Illuminate\Support\Carbon $updated_at
 * @property-read \Illuminate\Database\Eloquent\Collection<int, Asset> $assets
 * @property-read \Illuminate\Database\Eloquent\Collection<int, AssetFolder> $folders
 * @property-read \Illuminate\Database\Eloquent\Collection<int, MediaNamespaceClaim> $claims
 * @property-read \Illuminate\Database\Eloquent\Collection<int, AssetFolder> $rootFolders
 * @property-read \Illuminate\Database\Eloquent\Collection<int, Asset> $rootAssets
 */
class MediaNamespace extends Model
{
    /**
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'slug',
        'disk',
    ];

    /**
     * @return HasMany<Asset, $this>
     */
    public function assets(): HasMany
    {
        return $this->hasMany(Asset::class, 'media_namespace_id');
    }

    /**
     * @return HasMany<AssetFolder, $this>
     */
    public function folders(): HasMany
    {
        return $this->hasMany(AssetFolder::class, 'media_namespace_id');
    }

    /**
     * @return HasMany<MediaNamespaceClaim, $this>
     */
    public function claims(): HasMany
    {
        return $this->hasMany(MediaNamespaceClaim::class, 'media_namespace_id');
    }

    /**
     * @return HasMany<AssetFolder, $this>
     */
    public function rootFolders(): HasMany
    {
        return $this->folders()->whereNull('parent_id');
    }

    /**
     * @return HasMany<Asset, $this>
     */
    public function rootAssets(): HasMany
    {
        return $this->assets()->whereNull('asset_folder_id');
    }

    public function hasModelAccess(string $modelType): bool
    {
        return $this->claims()
            ->where('model_type', $modelType)
            ->exists();
    }
}
