<?php

declare(strict_types=1);

namespace MSML\MediaManager\DataObjects;

use DateTimeInterface;
use Spatie\LaravelData\Data;
use Illuminate\Support\Number;
use Spatie\LaravelData\Attributes\MapName;
use Spatie\LaravelData\Attributes\Computed;
use Spatie\LaravelData\Mappers\SnakeCaseMapper;
use MSML\MediaManager\Enums\MediaBrowserItemTypeEnum;

#[MapName(SnakeCaseMapper::class)]
final class MediaBrowserItem extends Data
{
    #[Computed]
    public string|null $formattedSize;

    #[Computed]
    public string|null $resolution;

    /**
     * @param  array<string, mixed>  $metadata
     * @param  array<string>|string|null  $path
     */
    public function __construct(
        public readonly int $id,
        public readonly string $name,
        public readonly MediaBrowserItemTypeEnum $type,
        public readonly array|string|null $path,
        public readonly int|null $size,
        public readonly DateTimeInterface $createdAt,
        public readonly string|null $thumbUrl,
        public readonly string|null $url,
        public readonly string|null $mimeType = null,
        public readonly array $metadata = [],
        public readonly int|null $width = null,
        public readonly int|null $height = null,
    ) {
        $this->formattedSize = $this->formattedSize();
        $this->resolution = $this->resolution();
    }

    private function formattedSize(): string|null
    {
        return $this->size ? Number::fileSize($this->size) : null;
    }

    private function resolution(): string|null
    {
        return $this->width && $this->height ? "{$this->width}x{$this->height}" : null;
    }
}
