<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use InvalidArgumentException;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;
use MSML\MediaManager\Models\MediaNamespace;

final class MoveAssetAction
{
    public function __construct(
        private readonly ResolveFolderFromPathAction $resolveFolderFromPathAction
    ) {}

    /**
     * @throws InvalidArgumentException
     */
    public function handle(Asset $asset, string $targetNamespaceSlug, string|null $targetFolderPath = null): Asset
    {
        return DB::transaction(function () use ($asset, $targetNamespaceSlug, $targetFolderPath) {
            $targetNamespace = MediaNamespace::whereSlug($targetNamespaceSlug)->firstOrFail();

            $targetFolder = null;
            if ($targetFolderPath !== null) {
                $targetFolder = $this->resolveFolderFromPathAction->handle($targetNamespace, $targetFolderPath);
            }

            $media = $asset->getFirstMediaItem();
            if (!$media) {
                throw new InvalidArgumentException('Asset has no media file to move');
            }

            if ($asset->media_namespace_id !== $targetNamespace->id) {
                $media->move(
                    model: $asset,
                    collectionName: $targetNamespace->name,
                    diskName: $targetNamespace->disk ?? '',
                    fileName: $media->file_name
                );
            }

            $asset->update([
                'media_namespace_id' => $targetNamespace->id,
                'asset_folder_id'    => $targetFolder?->id,
            ]);

            $asset->refresh();

            return $asset;
        });
    }
}
