<?php

declare(strict_types=1);

namespace MSML\MediaManager\Rules;

use Illuminate\Support\Number;
use MSML\MediaManager\Enums\AssetAction;

final class MaxFileSizeRule implements Rule
{
    public function __construct(
        private readonly int|null $maxSizeInBytes
    ) {}

    public function evaluate(RuleContext $context): RuleResult
    {
        return match ($context->action) {
            AssetAction::UPLOAD => $this->evaluateUpload($context),
            default             => RuleResult::skip(__('media-manager::rules.general.not_applicable'))
        };
    }

    private function evaluateUpload(RuleContext $context): RuleResult
    {
        if ($this->maxSizeInBytes === null) {
            return RuleResult::skip(
                __('media-manager::rules.max_file_size.skip_no_limit')
            );
        }

        if ($context->fileSize === null) {
            return RuleResult::skip(
                __('media-manager::rules.max_file_size.skip_no_info')
            );
        }

        if ($context->fileSize <= $this->maxSizeInBytes) {
            return RuleResult::pass();
        }

        return RuleResult::fail(
            __('media-manager::rules.max_file_size.fail', [
                'max_size'    => Number::fileSize($this->maxSizeInBytes),
                'actual_size' => Number::fileSize($context->fileSize),
            ]),
            [
                'max_size'    => $this->maxSizeInBytes,
                'actual_size' => $context->fileSize,
                'difference'  => $context->fileSize - $this->maxSizeInBytes,
            ]
        );
    }
}
