<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use Exception;
use InvalidArgumentException;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;

final class BulkDuplicateAssetsAction
{
    public function __construct(
        private readonly DuplicateAssetAction $duplicateAssetAction,
        private readonly ResolveAssetsAction $resolveAssetsAction
    ) {}

    /**
     * Duplicate multiple assets to a target folder.
     *
     * @param  array<int|Asset>  $assets  Array of Asset instances or asset IDs
     * @param  string|null  $targetFolderPath  Target folder path (null to keep in same folder)
     * @return array{duplicated: array<Asset>, failed: array<int, array{asset: Asset, error: string}>}
     */
    public function handle(
        array $assets,
        string|null $targetFolderPath = null
    ): array {
        if (empty($assets)) {
            return ['duplicated' => [], 'failed' => []];
        }

        return DB::transaction(function () use ($assets, $targetFolderPath) {
            $duplicated = [];
            $failed = [];
            $assetCollection = $this->resolveAssetsAction->handle($assets);

            foreach ($assetCollection as $asset) {
                try {
                    $duplicatedAsset = $this->duplicateAssetAction->handle(
                        $asset,
                        $targetFolderPath
                    );
                    $duplicated[] = $duplicatedAsset;
                } catch (Exception $e) {
                    $failed[] = [
                        'asset' => $asset,
                        'error' => $e->getMessage(),
                    ];
                }
            }

            if (count($failed) === count($assetCollection) && count($assetCollection) > 0) {
                throw new InvalidArgumentException('All asset duplications failed');
            }

            return [
                'duplicated' => $duplicated,
                'failed'     => $failed,
            ];
        });
    }
}
