<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\AssetFolder;
use MSML\MediaManager\Models\MediaNamespace;
use Illuminate\Validation\ValidationException;

final class MoveFolderAction
{
    public function __construct(
        private readonly ResolveFolderFromPathAction $resolveFolderFromPathAction
    ) {}

    /**
     * @param  int|AssetFolder  $folder  The folder to move
     * @param  string|null  $targetNamespaceSlug  The target namespace slug (null to stay in same namespace)
     * @param  string|null  $targetPath  The target parent folder path (null for root)
     *
     * @throws ValidationException
     */
    public function handle(AssetFolder|int $folder, string|null $targetNamespaceSlug = null, string|null $targetPath = null): AssetFolder
    {
        return DB::transaction(function () use ($folder, $targetNamespaceSlug, $targetPath) {
            if (!$folder instanceof AssetFolder) {
                $folder = AssetFolder::findOrFail($folder);
            }

            $targetNamespace = $folder->namespace;
            if ($targetNamespaceSlug !== null) {
                $targetNamespace = MediaNamespace::whereSlug($targetNamespaceSlug)->firstOrFail();
            }

            $targetParent = $this->resolveFolderFromPathAction->handle($targetNamespace, $targetPath);

            if ($folder->media_namespace_id === $targetNamespace->id && $folder->parent_id === $targetParent?->id) {
                return $folder;
            }

            if ($targetParent && $targetParent->id === $folder->id) {
                throw ValidationException::withMessages([
                    'target' => ['Cannot move a folder into itself'],
                ]);
            }

            if ($folder->media_namespace_id === $targetNamespace->id && $targetParent && $this->isDescendantOf($targetParent, $folder)) {
                throw ValidationException::withMessages([
                    'target' => ['Cannot move a folder into one of its own subfolders'],
                ]);
            }

            $existingFolder = AssetFolder::where('media_namespace_id', $targetNamespace->id)
                ->where('parent_id', $targetParent?->id)
                ->where('name', $folder->name)
                ->where('id', '!=', $folder->id)
                ->exists();

            if ($existingFolder) {
                throw ValidationException::withMessages([
                    'name' => ['A folder with this name already exists at the target location'],
                ]);
            }

            $isMovingAcrossNamespaces = $folder->media_namespace_id !== $targetNamespace->id;

            $folder->media_namespace_id = $targetNamespace->id;
            $folder->parent_id = $targetParent?->id;
            $folder->save();

            if ($isMovingAcrossNamespaces) {
                $this->updateDescendantNamespaces($folder, $targetNamespace->id);
            }

            /** @var AssetFolder */
            return $folder->fresh(['parent', 'children', 'namespace']);
        });
    }

    private function isDescendantOf(AssetFolder $potentialDescendant, AssetFolder $potentialAncestor): bool
    {
        $current = $potentialDescendant;

        while ($current->parent_id !== null) {
            if ($current->parent_id === $potentialAncestor->id) {
                return true;
            }
            /** @var AssetFolder $parent */
            $parent = $current->parent;
            $current = $parent;
        }

        return false;
    }

    private function updateDescendantNamespaces(AssetFolder $folder, int $namespaceId): void
    {
        $descendants = $folder->getAllDescendants();

        if ($descendants->isEmpty()) {
            return;
        }

        AssetFolder::whereIn('id', $descendants->pluck('id'))
            ->update(['media_namespace_id' => $namespaceId]);
    }
}
