<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\AssetFolder;
use Illuminate\Validation\ValidationException;

final class RenameFolderAction
{
    /**
     * @param  int|AssetFolder  $folder  The folder to rename
     * @param  string  $newName  The new name for the folder
     *
     * @throws ValidationException
     */
    public function handle(AssetFolder|int $folder, string $newName): AssetFolder
    {
        return DB::transaction(function () use ($folder, $newName) {
            if (!$folder instanceof AssetFolder) {
                $folder = AssetFolder::findOrFail($folder);
            }

            $newName = trim($newName);

            if (empty($newName)) {
                throw ValidationException::withMessages([
                    'name' => ['Folder name cannot be empty'],
                ]);
            }

            $existingFolder = AssetFolder::where('media_namespace_id', $folder->media_namespace_id)
                ->where('parent_id', $folder->parent_id)
                ->where('name', $newName)
                ->where('id', '!=', $folder->id)
                ->exists();

            if ($existingFolder) {
                throw ValidationException::withMessages([
                    'name' => ['A folder with this name already exists at this location'],
                ]);
            }

            $folder->name = $newName;
            $folder->save();

            /** @var AssetFolder */
            return $folder->fresh();
        });
    }
}
