<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use InvalidArgumentException;
use MSML\MediaManager\Models\AssetFolder;
use MSML\MediaManager\Models\MediaNamespace;

final class ResolveFolderFromPathAction
{
    /**
     * @throws InvalidArgumentException If the path cannot be resolved
     */
    public function handle(MediaNamespace $namespace, string|null $folderPath): AssetFolder|null
    {
        if (!$folderPath || trim($folderPath) === '') {
            return null;
        }

        $pathSegments = explode('/', trim($folderPath, '/'));
        $pathSegments = array_filter($pathSegments);

        if (empty($pathSegments)) {
            return null;
        }

        return $this->resolveFolder($namespace, $pathSegments);
    }

    /**
     * @param  array<string>  $pathSegments
     *
     * @throws InvalidArgumentException If any segment cannot be resolved
     */
    private function resolveFolder(MediaNamespace $namespace, array $pathSegments, AssetFolder|null $parent = null): AssetFolder
    {
        if (empty($pathSegments)) {
            throw new InvalidArgumentException('No path segments provided');
        }

        $currentSegment = array_shift($pathSegments);

        $folder = AssetFolder::where('media_namespace_id', $namespace->id)
            ->where('parent_id', $parent?->id)
            ->where('name', $currentSegment)
            ->firstOrFail();

        if (!empty($pathSegments)) {
            return $this->resolveFolder($namespace, $pathSegments, $folder);
        }

        return $folder;
    }
}
