<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use InvalidArgumentException;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;

final class BulkMoveAssetsAction
{
    public function __construct(
        private readonly MoveAssetAction $moveAssetAction,
        private readonly ResolveAssetsAction $resolveAssetsAction
    ) {}

    /**
     * Move multiple assets to a target namespace and optional folder.
     *
     * @param  array<int|Asset>  $assets  Array of Asset instances or asset IDs
     * @param  string  $targetNamespaceSlug  Target namespace slug
     * @param  string|null  $targetFolderPath  Optional target folder path
     * @return array{moved: array<Asset>, failed: array<int, array{asset: Asset, error: string}>}
     */
    public function handle(array $assets, string $targetNamespaceSlug, string|null $targetFolderPath = null): array
    {
        if (empty($assets)) {
            return ['moved' => [], 'failed' => []];
        }

        return DB::transaction(function () use ($assets, $targetNamespaceSlug, $targetFolderPath) {
            $moved = [];
            $failed = [];
            $assetCollection = $this->resolveAssetsAction->handle($assets);

            foreach ($assetCollection as $asset) {
                try {
                    $movedAsset = $this->moveAssetAction->handle($asset, $targetNamespaceSlug, $targetFolderPath);
                    $moved[] = $movedAsset;
                } catch (\Exception $e) {
                    $failed[] = [
                        'asset' => $asset,
                        'error' => $e->getMessage(),
                    ];
                }
            }

            if (count($failed) === count($assetCollection) && count($assetCollection) > 0) {
                throw new InvalidArgumentException('All asset moves failed');
            }

            return [
                'moved'  => $moved,
                'failed' => $failed,
            ];
        });
    }
}
