<?php

declare(strict_types=1);

namespace MSML\MediaManager\Rules;

use MSML\MediaManager\Enums\AssetAction;

final class MaxFilesRule implements Rule
{
    public function __construct(
        private readonly int|null $maxFiles
    ) {}

    public function evaluate(RuleContext $context): RuleResult
    {
        return match ($context->action) {
            AssetAction::UPLOAD => $this->evaluateUpload($context),
            default             => RuleResult::skip(__('media-manager::rules.general.not_applicable'))
        };
    }

    private function evaluateUpload(RuleContext $context): RuleResult
    {
        if ($this->maxFiles === null) {
            return RuleResult::skip(
                __('media-manager::rules.max_files.skip_no_limit')
            );
        }

        if ($context->currentFileCount >= $this->maxFiles) {
            return RuleResult::fail(
                __('media-manager::rules.max_files.fail', [
                    'max_files'     => $this->maxFiles,
                    'current_count' => $context->currentFileCount,
                ]),
                [
                    'max_files'     => $this->maxFiles,
                    'current_count' => $context->currentFileCount,
                ]
            );
        }

        return RuleResult::pass();
    }
}
