<?php

declare(strict_types=1);

namespace MSML\MediaManager\Rules;

use MSML\MediaManager\Enums\AssetAction;

final class AllowedMimeTypesRule implements Rule
{
    /**
     * @var string[]
     */
    private readonly array $allowedMimeTypes;

    /**
     * @param  string[]  $allowedMimeTypes
     */
    public function __construct(array $allowedMimeTypes)
    {
        $this->allowedMimeTypes = array_map('strval', $allowedMimeTypes);
    }

    public function evaluate(RuleContext $context): RuleResult
    {
        return match ($context->action) {
            AssetAction::UPLOAD => $this->evaluateUpload($context),
            default             => RuleResult::skip(__('media-manager::rules.general.not_applicable'))
        };
    }

    private function evaluateUpload(RuleContext $context): RuleResult
    {
        if (empty($this->allowedMimeTypes)) {
            return RuleResult::skip(
                __('media-manager::rules.allowed_mime_types.skip_no_restrictions')
            );
        }

        if ($context->mimeType === null) {
            return RuleResult::skip(
                __('media-manager::rules.allowed_mime_types.skip_no_info')
            );
        }

        if (in_array($context->mimeType, $this->allowedMimeTypes, true)) {
            return RuleResult::pass();
        }

        return RuleResult::fail(
            __('media-manager::rules.allowed_mime_types.fail', [
                'mime_type'     => $context->mimeType,
                'allowed_types' => implode(', ', $this->allowedMimeTypes),
            ]),
            [
                'mime_type'     => $context->mimeType,
                'allowed_types' => $this->allowedMimeTypes,
            ]
        );
    }
}
