<?php

declare(strict_types=1);

namespace MSML\MediaManager\Rules;

use Illuminate\Http\UploadedFile;
use Illuminate\Database\Eloquent\Model;
use MSML\MediaManager\Enums\AssetAction;
use MSML\MediaManager\Models\AssetFolder;
use MSML\MediaManager\Models\MediaNamespace;

final class RuleContext
{
    public function __construct(
        public readonly UploadedFile|null $file = null,
        public readonly int|null $fileSize = null,
        public readonly string|null $mimeType = null,
        public readonly string|null $filename = null,
        public readonly MediaNamespace|null $namespace = null,
        public readonly AssetFolder|null $folder = null,
        public readonly bool $inRoot = false,
        public readonly int $currentFileCount = 0,
        public readonly Model|null $model = null,
        public readonly AssetAction|null $action = null,
    ) {}

    public static function forUpload(
        UploadedFile $file,
        MediaNamespace $namespace,
        AssetFolder|null $folder = null,
        Model|null $model = null
    ): self {
        return new self(
            file: $file,
            fileSize: $file->getSize(),
            mimeType: $file->getMimeType(),
            filename: $file->getClientOriginalName(),
            namespace: $namespace,
            folder: $folder,
            inRoot: $folder === null,
            currentFileCount: $namespace->assets()->count(),
            model: $model,
            action: AssetAction::UPLOAD,
        );
    }

    public function getDisk(): string
    {
        if ($this->namespace?->disk === null) {
            throw new \RuntimeException(
                'Namespace does not have a disk configured.'
            );
        }

        return $this->namespace->disk;
    }
}
