<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use Error;
use ZipArchive;
use InvalidArgumentException;
use MSML\MediaManager\Models\Asset;
use Symfony\Component\HttpFoundation\StreamedResponse;

final class BulkDownloadAssetsAction
{
    public function __construct(
        private readonly ResolveAssetsAction $resolveAssetsAction
    ) {}

    /**
     * Download multiple assets as a zip file.
     *
     * @param  array<int|Asset>  $assets  Array of Asset instances or asset IDs
     * @param  string  $zipFilename  The name of the zip file to download
     *
     * @throws InvalidArgumentException
     */
    public function handle(array $assets, string $zipFilename = 'assets.zip'): StreamedResponse
    {
        if (empty($assets)) {
            throw new InvalidArgumentException('No assets provided for download');
        }

        $assetCollection = $this->resolveAssetsAction->handle($assets);

        if ($assetCollection->isEmpty()) {
            throw new InvalidArgumentException('No valid assets found for download');
        }

        $tempPath = tempnam(sys_get_temp_dir(), 'bulk_download_');
        $zip = new ZipArchive;

        if ($zip->open($tempPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new InvalidArgumentException('Could not create zip file');
        }

        $addedFiles = [];
        $filenameCounts = [];

        foreach ($assetCollection as $asset) {
            $media = $asset->getFirstMediaItem();

            if (!$media) {
                continue;
            }

            $filename = $media->file_name;
            $baseName = pathinfo($filename, PATHINFO_FILENAME);
            $extension = pathinfo($filename, PATHINFO_EXTENSION);

            if (isset($filenameCounts[$filename])) {
                $filenameCounts[$filename]++;
                $filename = $baseName . '_' . $filenameCounts[$filename] . '.' . $extension;
            } else {
                $filenameCounts[$filename] = 0;
            }

            try {
                $stream = $media->stream();

                if ($stream === null || !is_resource($stream)) {
                    throw new Error('media stream failed to open');
                }

                $fileContent = stream_get_contents($stream);
                fclose($stream);

                if ($fileContent) {
                    $zip->addFromString($filename, $fileContent);
                    $addedFiles[] = $filename;
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        $zip->close();

        if (empty($addedFiles)) {
            if (file_exists($tempPath)) {
                unlink($tempPath);
            }

            throw new InvalidArgumentException('No media files found in the selected assets');
        }

        return response()->streamDownload(function () use ($tempPath) {
            readfile($tempPath);
            unlink($tempPath);
        }, $zipFilename, [
            'Content-Type' => 'application/zip',
        ]);
    }
}
