<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\AssetFolder;

final class DeleteFolderAction
{
    public function __construct(
        private readonly DeleteAssetAction $deleteAssetAction
    ) {}

    /**
     * @param  int|AssetFolder  $folder  The folder to delete
     * @return array{folders: int, assets: int} Count of deleted folders and assets
     */
    public function handle(AssetFolder|int $folder): array
    {
        return DB::transaction(function () use ($folder) {
            if (!$folder instanceof AssetFolder) {
                $folder = AssetFolder::findOrFail($folder);
            }

            ['folders' => $deletedFolders, 'assets' => $deletedAssets] = $this->deleteContents($folder);

            $folder->delete();
            $deletedFolders++;

            return [
                'folders' => $deletedFolders,
                'assets'  => $deletedAssets,
            ];
        });
    }

    /**
     * @return array{folders: int, assets: int}
     */
    private function deleteContents(AssetFolder $folder): array
    {
        $deletedFolders = 0;
        $deletedAssets = 0;

        $folder->loadMissing(['assets', 'children']);

        foreach ($folder->assets as $asset) {
            $this->deleteAssetAction->handle($asset);
            $deletedAssets++;
        }

        foreach ($folder->children as $childFolder) {
            ['folders' => $childFolders, 'assets' => $childAssets] = $this->deleteContents($childFolder);
            $deletedFolders += $childFolders;
            $deletedAssets += $childAssets;

            $childFolder->delete();
            $deletedFolders++;
        }

        return [
            'folders' => $deletedFolders,
            'assets'  => $deletedAssets,
        ];
    }
}
