<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use MSML\MediaManager\Models\AssetFolder;
use MSML\MediaManager\Models\MediaNamespace;

final class GetValidMoveFolderTargetsAction
{
    /**
     * @param  int|AssetFolder  $folderToMove  The folder that will be moved
     * @param  bool  $includeAllNamespaces  Whether to include folders from all namespaces
     * @return array<int, array{value: string|null, label: string, folder_id: int|null, namespace_id: int, namespace_slug: string, depth: int}>
     */
    public function handle(AssetFolder|int $folderToMove, bool $includeAllNamespaces = false): array
    {
        if (!$folderToMove instanceof AssetFolder) {
            $folderToMove = AssetFolder::findOrFail($folderToMove);
        }

        $excludedIds = $this->getExcludedFolderIds($folderToMove);
        $options = [];

        if ($includeAllNamespaces) {
            $namespaces = MediaNamespace::all();

            foreach ($namespaces as $namespace) {
                $options[] = [
                    'value'          => null,
                    'label'          => $namespace->name . ' (Root)',
                    'folder_id'      => null,
                    'namespace_id'   => $namespace->id,
                    'namespace_slug' => $namespace->slug,
                    'depth'          => 0,
                ];

                $query = AssetFolder::where('media_namespace_id', $namespace->id)
                    ->with('parent');

                if ($namespace->id === $folderToMove->media_namespace_id) {
                    $query->whereNotIn('id', $excludedIds);
                }

                $allFolders = $query->get()->withRelationshipAutoloading();
                $rootFolders = $allFolders->whereNull('parent_id');

                foreach ($rootFolders as $folder) {
                    $options = array_merge($options, $this->buildFolderOptions($folder, $allFolders, 1, $namespace));
                }
            }
        } else {
            $folderToMove->loadMissing('namespace');
            $namespace = $folderToMove->namespace;

            $options[] = [
                'value'          => null,
                'label'          => $namespace->name . ' (Root)',
                'folder_id'      => null,
                'namespace_id'   => $namespace->id,
                'namespace_slug' => $namespace->slug,
                'depth'          => 0,
            ];

            $allFolders = AssetFolder::where('media_namespace_id', $namespace->id)
                ->whereNotIn('id', $excludedIds)
                ->with('parent')
                ->get()
                ->withRelationshipAutoloading();

            $rootFolders = $allFolders->whereNull('parent_id');

            foreach ($rootFolders as $folder) {
                $options = array_merge($options, $this->buildFolderOptions($folder, $allFolders, 1, $namespace));
            }
        }

        return $options;
    }

    /**
     * @return list<int>
     */
    private function getExcludedFolderIds(AssetFolder $folder): array
    {
        $excludedIds = [$folder->id];
        $folder->loadMissing('children');
        /** @var list<int> $descendantIds */
        $descendantIds = $folder->getAllDescendants()->pluck('id')->values()->toArray();

        return array_merge($excludedIds, $descendantIds);
    }

    /**
     * @param  \Illuminate\Support\Collection<int, AssetFolder>  $allFolders
     * @return list<array{value: string, label: string, folder_id: int, namespace_id: int, namespace_slug: string, depth: int}>
     */
    private function buildFolderOptions(AssetFolder $folder, $allFolders, int $depth = 0, MediaNamespace|null $namespace = null): array
    {
        $pathSegments = $folder->getPathSegments();
        $ns = $namespace ?? $folder->namespace;

        $options = [[
            'value'          => implode('/', $pathSegments),
            'label'          => implode('/', $pathSegments),
            'folder_id'      => $folder->id,
            'namespace_id'   => $ns->id,
            'namespace_slug' => $ns->slug,
            'depth'          => $depth,
        ]];

        $children = $allFolders->where('parent_id', $folder->id);

        foreach ($children as $child) {
            $options = array_merge($options, $this->buildFolderOptions($child, $allFolders, $depth + 1, $namespace));
        }

        return $options;
    }
}
