<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use InvalidArgumentException;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;
use MSML\MediaManager\Rules\RuleContext;
use MSML\MediaManager\Facades\MediaManager;
use MSML\MediaManager\Models\MediaNamespace;
use Illuminate\Validation\ValidationException;

final class UploadFileAction
{
    /**
     * @var array{width: int, height: int, fit: string, quality: int, format: string|null}
     */
    private const DEFAULT_THUMBNAIL_CONVERSION_SETTINGS = [
        'width'   => 300,
        'height'  => 300,
        'fit'     => 'crop',
        'quality' => 80,
        'format'  => null,
    ];

    public function __construct(
        private readonly ResolveFolderFromPathAction $resolveFolderFromPathAction
    ) {}

    /**
     * @param  array{width: int, height: int, fit: string, quality: int, format: string|null}|null  $thumbnailConversionSettings
     *
     * @throws InvalidArgumentException
     */
    public function handle(
        string $namespaceSlug,
        UploadedFile $file,
        string|null $folderPath = null,
        array|null $thumbnailConversionSettings = null
    ): Asset {
        return DB::transaction(function () use ($namespaceSlug, $file, $folderPath, $thumbnailConversionSettings) {
            $namespace = MediaNamespace::whereSlug($namespaceSlug)->firstOrFail();

            $folder = null;

            if (!$namespace->disk) {
                throw new InvalidArgumentException('No disk configured');
            }

            $namespaceConfiguration = MediaManager::namespaceConfigurationFor($namespaceSlug);

            if ($namespaceConfiguration) {
                $context = RuleContext::forUpload($file, $namespace);

                $result = $namespaceConfiguration->validateAction($context);

                if (!$result->passed()) {
                    throw ValidationException::withMessages([
                        'file' => [$result->getFirstError()],
                    ]);
                }
            }

            $folder = $this->resolveFolderFromPathAction->handle($namespace, $folderPath);

            $metadata = [
                'original_name' => $file->getClientOriginalName(),
            ];

            if (str_starts_with($file->getMimeType() ?? '', 'image/')) {
                $imageSize = getimagesize($file->getPathname());
                if ($imageSize !== false) {
                    $metadata['width'] = $imageSize[0];
                    $metadata['height'] = $imageSize[1];
                }

                $conversions = $namespaceConfiguration?->getConversions() ?? [];

                if ($thumbnailConversionSettings !== null) {
                    $conversions['thumb'] = $thumbnailConversionSettings;
                }

                if ($conversions === []) {
                    $conversions['thumb'] = self::DEFAULT_THUMBNAIL_CONVERSION_SETTINGS;
                }

                $metadata['conversions'] = $conversions;
            }

            /** @var Asset */
            $asset = Asset::create([
                'media_namespace_id'  => $namespace->id,
                'asset_folder_id'     => $folder?->id,
                'uploaded_by_user_id' => auth()->id(),
                'metadata'            => $metadata,
            ]);

            $asset->addMedia($file)
                ->toMediaCollection($namespace->name, $namespace->disk);

            return $asset;
        });
    }
}
