<?php

declare(strict_types=1);

namespace MSML\MediaManager\Models;

use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use Illuminate\Database\Eloquent\Model;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\Conversions\Conversion;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

/**
 * @property int $id
 * @property int $media_namespace_id
 * @property int|null $asset_folder_id
 * @property int|null $uploaded_by_user_id
 * @property array{
 *     conversions?: array<string, array{
 *         fit: string,
 *         width: int,
 *         height: int,
 *         format: string|null,
 *         quality: int
 *     }>,
 *     height: int,
 *     width: int,
 *     alt_text?: string,
 *     original_name?: string
 * }|null $metadata
 * @property \Illuminate\Support\Carbon $created_at
 * @property \Illuminate\Support\Carbon $updated_at
 * @property-read MediaNamespace $namespace
 * @property-read AssetFolder|null $folder
 * @property-read \Illuminate\Database\Eloquent\Model|null $uploadedByUser
 * @property-read string      $url
 * @property-read string      $thumb_url
 * @property-read string|null $original_name
 * @property-read string      $extension
 * @property-read string|null $mime_type
 * @property-read int|null    $size
 * @property-read string      $disk
 * @property-read string      $full_path
 * @property-read int|null    $width
 * @property-read int|null    $height
 * @property-read \Illuminate\Database\Eloquent\Collection<int, Model> $assetables
 *
 * @method Conversion addMediaConversion(string $name)
 *
 * @mixin \Eloquent
 */
class Asset extends Model implements HasMedia
{
    use InteractsWithMedia;

    public bool $registerMediaConversionsUsingModelInstance = true;

    /**
     * @var list<string>
     */
    protected $fillable = [
        'media_namespace_id',
        'asset_folder_id',
        'uploaded_by_user_id',
        'metadata',
    ];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'metadata' => 'array',
        ];
    }

    /**
     * @return BelongsTo<MediaNamespace, $this>
     */
    public function namespace(): BelongsTo
    {
        return $this->belongsTo(MediaNamespace::class, 'media_namespace_id');
    }

    /**
     * @return BelongsTo<AssetFolder, $this>
     */
    public function folder(): BelongsTo
    {
        return $this->belongsTo(AssetFolder::class, 'asset_folder_id');
    }

    /**
     * @return BelongsTo<Model, $this>
     */
    public function uploadedByUser(): BelongsTo
    {
        /** @var class-string<Model> $userModel */
        $userModel = config('media-manager.user_model', 'App\\Models\\User');

        return $this->belongsTo($userModel, 'uploaded_by_user_id');
    }

    /**
     * @return MorphToMany<Model, $this>
     */
    public function assetables(): MorphToMany
    {
        return $this->morphToMany(Model::class, 'assetable')
            ->withPivot(['collection', 'order'])
            ->withTimestamps();
    }

    public function getFirstMediaItem(): Media|null
    {
        if ($this->exists) {
            $this->loadMissing('media');
        }

        return $this->media->first();
    }

    public function getDefaultCollectionName(): string
    {
        if ($this->exists) {
            $this->loadMissing('namespace');
        }

        return $this->namespace->name;
    }

    public function getFirstMediaUrl(string|null $collection = null, string $conversion = ''): string|null
    {
        $collectionName = $collection ?? $this->getDefaultCollectionName();

        $media = $this->getFirstMedia($collectionName);

        if (!$media) {
            return null;
        }

        if ($conversion) {
            return $media->hasGeneratedConversion($conversion) ? $media->getUrl($conversion) : null;
        }

        return $media->getUrl();
    }

    public function getConversionUrl(string $conversion): string|null
    {
        return $this->getFirstMediaUrl(null, $conversion);
    }

    public function hasConversion(string $conversion): bool
    {
        $media = $this->getFirstMediaItem();

        if (!$media) {
            return false;
        }

        return $media->hasGeneratedConversion($conversion);
    }

    public function getPath(): string
    {
        if ($this->exists) {
            $this->loadMissing(['folder', 'namespace']);
        }

        if ($this->folder) {
            return $this->folder->path;
        }

        return $this->namespace->slug;
    }

    /**
     * @return Attribute<string|null,void>
     */
    protected function url(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMediaUrl());
    }

    /**
     * @return Attribute<string|null,void>
     */
    protected function thumbUrl(): Attribute
    {
        return Attribute::get(fn () => $this->getConversionUrl('thumb') ?? $this->url);
    }

    /**
     * @return Attribute<string|null,void>
     */
    protected function originalName(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMediaItem()?->file_name);
    }

    /**
     * @return Attribute<string,void>
     */
    protected function extension(): Attribute
    {
        return Attribute::get(fn () => $this->original_name ? pathinfo($this->original_name, PATHINFO_EXTENSION) : '');
    }

    /**
     * @return Attribute<string|null,void>
     */
    protected function mimeType(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMediaItem()?->mime_type);
    }

    /**
     * @return Attribute<int|null,never>
     */
    protected function size(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMediaItem()?->size);
    }

    /**
     * @return Attribute<string|null,never>
     */
    protected function disk(): Attribute
    {
        return Attribute::get(function () {
            if ($this->exists) {
                $this->loadMissing('namespace');
            }

            return $this->getFirstMediaItem()->disk ?? $this->namespace->disk;
        });
    }

    /**
     * @return Attribute<non-falsy-string,void>
     */
    protected function fullPath(): Attribute
    {
        return Attribute::get(function () {
            if ($this->exists) {
                $this->loadMissing(['folder', 'namespace']);
            }

            return $this->folder
                ? "{$this->folder->path}/{$this->original_name}"
                : "{$this->namespace->slug}/{$this->original_name}";
        });
    }

    public function registerMediaConversions(Media|null $media = null): void
    {
        $conversions = $this->getModelNamespaceConversions();

        foreach ($conversions as $name => $settings) {

            /** @var Conversion $conversion */
            $conversion = $this
                ->addMediaConversion($name)
                ->width($settings['width'])
                ->height($settings['height'])
                ->quality($settings['quality'])
                ->fit(Fit::from($settings['fit']), $settings['width'], $settings['height']);

            if (isset($settings['format'])) {
                $conversion->format($settings['format']);
            }

            $conversion->nonQueued();
        }

    }

    /**
     * @return array<string, array{
     *     width: int,
     *     height: int,
     *     fit: string,
     *     quality: int,
     *     format: string|null
     * }>
     */
    public function getModelNamespaceConversions(): array
    {
        if (!$this->metadata || !isset($this->metadata['conversions'])) {
            return [];
        }

        return $this->metadata['conversions'];
    }

    /**
     * @return Attribute<int|null,void>
     */
    protected function width(): Attribute
    {
        return Attribute::get(fn () => isset($this->metadata['width']) ? (int)$this->metadata['width'] : null);
    }

    /**
     * @return Attribute<int|null,void>
     */
    protected function height(): Attribute
    {
        return Attribute::get(fn () => isset($this->metadata['height']) ? (int)$this->metadata['height'] : null);
    }
}
