<?php

declare(strict_types=1);

namespace MSML\MediaManager\Rules;

use MSML\MediaManager\Enums\AssetAction;

final class ImageDimensionsRule implements Rule
{
    public function __construct(
        private readonly int $maxWidth,
        private readonly int $maxHeight
    ) {}

    public function evaluate(RuleContext $context): RuleResult
    {
        return match ($context->action) {
            AssetAction::UPLOAD => $this->evaluateUpload($context),
            default             => RuleResult::skip(__('media-manager::rules.general.not_applicable'))
        };
    }

    private function evaluateUpload(RuleContext $context): RuleResult
    {
        if ($context->file === null) {
            return RuleResult::skip(
                __('media-manager::rules.image_dimensions.skip_no_file')
            );
        }

        if (!str_starts_with($context->mimeType ?? '', 'image/')) {
            return RuleResult::skip(
                __('media-manager::rules.image_dimensions.skip_not_image')
            );
        }

        $imageSize = getimagesize($context->file->getPathname());

        if ($imageSize === false) {
            return RuleResult::fail(
                __('media-manager::rules.image_dimensions.fail_cannot_determine')
            );
        }

        [$width, $height] = $imageSize;

        if ($width <= $this->maxWidth && $height <= $this->maxHeight) {
            return RuleResult::pass();
        }

        return RuleResult::fail(
            __('media-manager::rules.image_dimensions.fail', [
                'max_width'     => $this->maxWidth,
                'max_height'    => $this->maxHeight,
                'actual_width'  => $width,
                'actual_height' => $height,
            ]),
            [
                'max_width'     => $this->maxWidth,
                'max_height'    => $this->maxHeight,
                'actual_width'  => $width,
                'actual_height' => $height,
            ]
        );
    }
}
