<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use InvalidArgumentException;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;
use Illuminate\Database\Eloquent\Builder;
use MSML\MediaManager\Models\AssetFolder;

final class DuplicateAssetAction
{
    public function __construct(
        private readonly ResolveFolderFromPathAction $resolveFolderFromPathAction
    ) {}

    /**
     * @throws InvalidArgumentException
     */
    public function handle(Asset $asset, string|null $targetFolderPath = null): Asset
    {
        return DB::transaction(function () use ($asset, $targetFolderPath) {
            $originalMedia = $asset->getFirstMediaItem();

            if (!$originalMedia) {
                throw new InvalidArgumentException('Asset has no media file to duplicate');
            }

            $targetFolder = null;
            if ($targetFolderPath !== null) {
                $targetFolder = $this->resolveFolderFromPathAction->handle($asset->namespace, $targetFolderPath);

                if ($targetFolder && $targetFolder->media_namespace_id !== $asset->media_namespace_id) {
                    throw new InvalidArgumentException('Cannot duplicate asset to folder in different namespace');
                }
            } else {
                $targetFolder = $asset->folder;
            }

            $metadata = $asset->metadata ?? [];
            $metadata['duplicated_from'] = $asset->id;
            $metadata['duplicated_at'] = now()->toISOString();

            $duplicatedAsset = Asset::create([
                'media_namespace_id' => $asset->media_namespace_id,
                'asset_folder_id'    => $targetFolder?->id,
                'metadata'           => $metadata,
            ]);

            $newFileName = $this->generateUniqueFileName($originalMedia->file_name, $targetFolder);

            $originalMedia->copy(
                model: $duplicatedAsset,
                collectionName: $asset->namespace->name,
                diskName: $asset->namespace->disk ?? '',
                fileName: $newFileName
            );

            return $duplicatedAsset;
        });
    }

    private function generateUniqueFileName(string $originalFileName, AssetFolder|null $targetFolder): string
    {
        $pathInfo = pathinfo($originalFileName);
        $baseName = $pathInfo['filename'];
        $extension = $pathInfo['extension'] ?? '';

        $counter = 1;
        $newFileName = $originalFileName;

        while ($this->fileNameExists($newFileName, $targetFolder)) {
            $newFileName = $baseName . '_copy_' . $counter . ($extension ? '.' . $extension : '');
            $counter++;
        }

        return $newFileName;
    }

    private function fileNameExists(string $fileName, AssetFolder|null $targetFolder): bool
    {

        $query = Asset::whereHas('media', function (Builder $query) use ($fileName) {
            /** @phpstan-ignore-next-line  - Spatie media library is not recognized as a model**/
            $query->where('file_name', $fileName);
        });

        if ($targetFolder) {
            $query->where('asset_folder_id', $targetFolder->id);
        } else {
            $query->whereNull('asset_folder_id');
        }

        return $query->exists();
    }
}
