<?php

declare(strict_types=1);

namespace MSML\MediaManager\Traits;

use InvalidArgumentException;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;
use MSML\MediaManager\Rules\RuleContext;
use MSML\MediaManager\Models\MediaNamespace;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use MSML\MediaManager\ValueObjects\NamespaceConfiguration;

/**
 * Trait HasAssets
 *
 * Provides media asset management functionality for Eloquent models.
 * Models using this trait can upload, organize, and manage media files
 * within namespaces and folders with proper permission control.
 */
trait HasAssets
{
    /**
     * Configure the media namespace for this model.
     *
     * This method must be implemented by models using this trait to define
     * the namespace configuration including name and rules.
     *
     * @return NamespaceConfiguration The namespace configuration
     */
    abstract public function getNamespaceConfiguration(): NamespaceConfiguration;

    /**
     * Define the polymorphic many-to-many relationship with assets.
     */
    public function assets(): MorphToMany
    {
        return $this->morphToMany(
            Asset::class,
            'assetable'
        )->withPivot(['collection', 'order'])
            ->withTimestamps();
    }

    /**
     * Attach a new asset to this model.
     *
     * Uploads a file to the model's namespace, optionally placing it in a specific folder.
     * The asset is linked to this model with collection and ordering information.
     *
     * @param  UploadedFile  $file  The file to upload
     * @param  int|null  $folderId  Optional folder ID to place the asset in
     * @param  string|null  $collection  Collection name for organizing assets (default: 'default')
     * @param  int  $order  Order position for the asset within its collection
     * @return Asset The created asset
     *
     * @throws InvalidArgumentException If namespace doesn't exist or model lacks access
     */
    public function attachAsset(
        UploadedFile $file,
        int|null $folderId = null,
        int $order = 0,
        string|null $collection = null
    ): Asset {
        $namespace = $this->resolveNamespace();
        $folder = $folderId ? $namespace->folders()->find($folderId) : null;
        $configuration = $this->getNamespaceConfiguration();

        $context = RuleContext::forUpload(
            file: $file,
            namespace: $namespace,
            folder: $folder,
            model: $this,
        );

        $result = $configuration->validateAction($context);

        if (!$result->passed()) {
            throw new ValidationException($result->getFirstError());
        }

        return DB::transaction(function () use ($namespace, $folderId, $file, $order, $collection) {
            $metadata = [
                'original_name' => $file->getClientOriginalName(),
                'uploaded_by'   => auth()->id(),
            ];

            if (str_starts_with($file->getMimeType() ?? '', 'image/')) {
                $imageSize = getimagesize($file->getPathname());
                if ($imageSize !== false) {
                    $metadata['width'] = $imageSize[0];
                    $metadata['height'] = $imageSize[1];
                }
            }

            /** @var Asset */
            $asset = Asset::create([
                'media_namespace_id' => $namespace->id,
                'asset_folder_id'    => $folderId,
                'metadata'           => $metadata,
            ]);

            $asset->addMedia($file)
                ->toMediaCollection($collection ?? $namespace->name, $namespace->disk);

            $this->assets()->attach($asset->id, [
                'order' => $order,
            ]);

            return $asset;
        });
    }

    /**
     * Delete an asset and remove its association with this model.
     *
     * This will remove the asset from storage, clear all media files,
     * and detach it from this model.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to delete
     * @return bool True if deletion was successful
     *
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If asset not found
     */
    public function deleteAsset(Asset|int $asset): bool
    {
        $asset = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);

        $this->assets()->detach($asset->id);

        $asset->clearMediaCollection();

        return $asset->delete();
    }

    /**
     * Move an asset to a different folder within the same namespace.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to move
     * @param  int|null  $targetFolderId  Target folder ID, or null for namespace root
     *
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If asset not found
     */
    public function moveAsset(Asset|int $asset, int|null $targetFolderId): void
    {
        $asset = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);

        $asset->update(['asset_folder_id' => $targetFolderId]);
    }

    /**
     * Download an asset as a streamed response.
     *
     * Returns the asset's media file as a downloadable response with
     * appropriate headers for the file type.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to download
     * @return StreamedResponse The download response
     *
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If asset not found
     * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException If media file not found
     */
    public function downloadAsset(Asset|int $asset): StreamedResponse
    {
        $asset = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);

        $media = $asset->getFirstMedia($asset->namespace->name);

        if (!$media) {
            abort(404, 'Media file not found');
        }

        return response()->streamDownload(function () use ($media) {
            echo $media->get();
        }, $media->file_name, [
            'Content-Type' => $media->mime_type,
        ]);
    }

    /**
     * Get the namespace configuration that applies to this model.
     *
     * Returns the model's namespace configuration with all rules.
     *
     * @return NamespaceConfiguration The namespace configuration for this model
     */
    public function getNamespaceRules(): NamespaceConfiguration
    {
        return $this->getNamespaceConfiguration();
    }

    /**
     * Resolve the media namespace for this model.
     *
     * Finds the namespace by slug and verifies the model has access to it.
     *
     * @return MediaNamespace The resolved namespace
     *
     * @throws InvalidArgumentException If namespace doesn't exist or model lacks access
     */
    protected function resolveNamespace(): MediaNamespace
    {
        $configuration = $this->getNamespaceConfiguration();
        $slug = $configuration->getName();

        $namespace = MediaNamespace::whereSlug($slug)->first();

        if (!$namespace) {
            throw new InvalidArgumentException("Namespace '{$slug}' does not exist");
        }

        if (!$namespace->hasModelAccess(static::class)) {
            throw new InvalidArgumentException('Model ' . static::class . " does not have access to namespace '{$slug}'");
        }

        return $namespace;
    }
}
