<?php

declare(strict_types=1);

namespace MSML\MediaManager\Traits;

use Illuminate\Support\Str;
use InvalidArgumentException;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;
use MSML\MediaManager\Rules\RuleContext;
use MSML\MediaManager\Models\MediaNamespace;
use Illuminate\Validation\ValidationException;
use MSML\MediaManager\Models\MediaNamespaceClaim;
use MSML\MediaManager\DataObjects\MediaBrowserItem;
use MSML\MediaManager\Enums\MediaBrowserItemTypeEnum;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use MSML\MediaManager\ValueObjects\NamespaceConfiguration;

/**
 * Trait HasAssets
 *
 * Provides media asset management functionality for Eloquent models.
 * Models using this trait can upload, organize, and manage media files
 * within namespaces and folders with proper permission control.
 */
trait HasAssets
{
    /**
     * Configure the media namespace for this model.
     *
     * This method must be implemented by models using this trait to define
     * the namespace configuration including name and rules.
     *
     * @return NamespaceConfiguration The namespace configuration
     */
    abstract public function getNamespaceConfiguration(): NamespaceConfiguration;

    /**
     * Define the polymorphic many-to-many relationship with assets.
     */
    public function assets(): MorphToMany
    {
        return $this->morphToMany(
            Asset::class,
            'assetable'
        )->withPivot(['collection', 'order'])
            ->withTimestamps();
    }

    /**
     * Attach a new asset to this model.
     *
     * Uploads a file to the model's namespace, optionally placing it in a specific folder.
     * The asset is linked to this model with collection and ordering information.
     *
     * @param  UploadedFile  $file  The file to upload
     * @param  int|null  $folderId  Optional folder ID to place the asset in
     * @param  string|null  $collection  Collection name for organizing assets (default: 'default')
     * @param  int  $order  Order position for the asset within its collection
     * @return Asset The created asset
     *
     * @throws InvalidArgumentException If namespace doesn't exist or model lacks access
     */
    public function attachAsset(
        UploadedFile $file,
        int|null $folderId = null,
        int $order = 0,
        string|null $collection = null
    ): Asset {
        $namespace = $this->resolveNamespace();
        $folder = $folderId ? $namespace->folders()->find($folderId) : null;
        $configuration = $this->getNamespaceConfiguration();

        $context = RuleContext::forUpload(
            file: $file,
            namespace: $namespace,
            folder: $folder,
            model: $this,
        );

        $result = $configuration->validateAction($context);

        if (!$result->passed()) {
            throw ValidationException::withMessages([
                'file' => [$result->getFirstError()],
            ]);
        }

        return DB::transaction(function () use ($namespace, $folderId, $file, $order, $collection, $configuration) {
            $metadata = [
                'original_name' => $file->getClientOriginalName(),
                'conversions'   => $configuration->getConversions(),
            ];

            if (str_starts_with($file->getMimeType() ?? '', 'image/')) {
                $imageSize = getimagesize($file->getPathname());
                if ($imageSize !== false) {
                    $metadata['width'] = $imageSize[0];
                    $metadata['height'] = $imageSize[1];
                }
            }

            /** @var Asset */
            $asset = Asset::create([
                'media_namespace_id'  => $namespace->id,
                'asset_folder_id'     => $folderId,
                'uploaded_by_user_id' => auth()->id(),
                'metadata'            => $metadata,
            ]);

            $asset->addMedia($file)
                ->toMediaCollection($collection ?? $namespace->name, $namespace->disk);

            $this->assets()->attach($asset->id, [
                'collection' => $collection ?? 'default',
                'order'      => $order,
            ]);

            return $asset;
        });
    }

    /**
     * Link an existing asset to this model.
     *
     * Links an existing asset to this model after validating the action through the rules engine.
     * The asset must exist within the model's namespace.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to link
     * @param  string|null  $collection  Collection name for organizing assets (default: 'default')
     * @param  int  $order  Order position for the asset within its collection
     * @return Asset The linked asset
     *
     * @throws InvalidArgumentException If asset doesn't exist in the model's namespace
     * @throws ValidationException If validation fails
     */
    public function linkAsset(
        Asset|int $asset,
        int $order = 0,
        string|null $collection = null
    ): Asset {
        $asset = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);
        $namespace = $this->resolveNamespace();

        if ($asset->media_namespace_id !== $namespace->id) {
            throw new InvalidArgumentException(
                "Asset does not belong to namespace '{$namespace->name}'"
            );
        }

        $configuration = $this->getNamespaceConfiguration();

        $context = RuleContext::forLink(
            asset: $asset,
            namespace: $namespace,
            folder: $asset->folder,
            model: $this,
        );

        $result = $configuration->validateAction($context);

        if (!$result->passed()) {
            throw ValidationException::withMessages([
                'file' => [$result->getFirstError()],
            ]);
        }

        $resolvedCollection = $collection ?? 'default';

        $existingLink = $this->assets()
            ->wherePivot('collection', $resolvedCollection)
            ->find($asset->id);

        if ($existingLink) {
            $this->assets()
                ->wherePivot('collection', $resolvedCollection)
                ->updateExistingPivot($asset->id, [
                    'order' => $order,
                ]);
        } else {
            $this->assets()->attach($asset->id, [
                'collection' => $resolvedCollection,
                'order'      => $order,
            ]);
        }

        return $asset->fresh();
    }

    /**
     * Delete an asset and remove its association with this model.
     *
     * This will remove the asset from storage, clear all media files,
     * and detach it from this model.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to delete
     * @return bool True if deletion was successful
     *
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If asset not found
     */
    public function deleteAsset(Asset|int $asset): bool
    {
        $asset = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);

        $this->assets()->detach($asset->id);

        $asset->clearMediaCollection();

        return $asset->delete();
    }

    /**
     * Download an asset as a streamed response.
     *
     * Returns the asset's media file as a downloadable response with
     * appropriate headers for the file type.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to download
     * @return StreamedResponse The download response
     *
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException If asset not found
     * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException If media file not found
     */
    public function downloadAsset(Asset|int $asset): StreamedResponse
    {
        $asset = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);

        $media = $asset->getFirstMedia($asset->namespace->name);

        if (!$media) {
            abort(404, 'Media file not found');
        }

        return response()->streamDownload(function () use ($media) {
            echo $media->get();
        }, $media->file_name, [
            'Content-Type' => $media->mime_type,
        ]);
    }

    /**
     * Get the namespace configuration that applies to this model.
     *
     * Returns the model's namespace configuration with all rules.
     *
     * @return NamespaceConfiguration The namespace configuration for this model
     */
    public function getNamespaceRules(): NamespaceConfiguration
    {
        return $this->getNamespaceConfiguration();
    }

    /**
     * @throws InvalidArgumentException If namespace doesn't exist and auto-create is disabled
     */
    protected function resolveNamespace(): MediaNamespace
    {
        $configuration = $this->getNamespaceConfiguration();
        $slug = $configuration->getName();

        $namespace = MediaNamespace::whereSlug($slug)->first();

        if (!$namespace) {
            if (config('media-manager.auto_create_namespaces.enabled', false)) {
                return $this->ensureNamespaceExists();
            }

            throw new InvalidArgumentException("Namespace '{$slug}' does not exist");
        }

        if (!$namespace->hasModelAccess(static::class)) {
            if (config('media-manager.auto_create_namespaces.create_claims', false)) {
                MediaNamespaceClaim::firstOrCreate([
                    'media_namespace_id' => $namespace->id,
                    'model_type'         => static::class,
                ]);
            } else {
                throw new InvalidArgumentException('Model ' . static::class . " does not have access to namespace '{$slug}'");
            }
        }

        return $namespace;
    }

    protected function ensureNamespaceExists(): MediaNamespace
    {
        $configuration = $this->getNamespaceConfiguration();
        $slug = Str::slug($configuration->getName(), ' ');
        $name = $configuration->getName();

        return DB::transaction(function () use ($slug, $name, $configuration) {
            $namespace = MediaNamespace::firstOrCreate(
                ['slug' => $slug],
                [
                    'name'   => $name,
                    'disk'   => $configuration->getDisk(),
                    'hidden' => $configuration->isHidden(),
                ]
            );

            if (config('media-manager.auto_create_namespaces.create_claims', true)) {
                MediaNamespaceClaim::firstOrCreate([
                    'media_namespace_id' => $namespace->id,
                    'model_type'         => static::class,
                ]);

            }

            return $namespace;
        });
    }

    /**
     * Get all assets for a specific collection.
     *
     * @param  string  $collection  The collection name
     * @return \Illuminate\Database\Eloquent\Collection<int, Asset>
     */
    public function getAssetsByCollection(string $collection): \Illuminate\Database\Eloquent\Collection
    {
        return $this->assets()
            ->with('namespace')
            ->wherePivot('collection', $collection)
            ->orderByPivot('order')
            ->get();
    }

    /**
     * @param  string  $collection  The collection name
     * @return \Illuminate\Support\Collection<int, MediaBrowserItem>
     */
    public function getMediaBrowserItemsByCollection(string $collection): \Illuminate\Support\Collection
    {
        return $this->getAssetsByCollection($collection)->map(fn (Asset $asset) => new MediaBrowserItem(
            id: $asset->id,
            name: $asset->original_name ?? 'Untitled',
            type: MediaBrowserItemTypeEnum::Asset,
            path: null,
            size: $asset->size,
            thumbUrl: $asset->thumb_url,
            url: $asset->url,
            createdAt: $asset->created_at,
            mimeType: $asset->mime_type,
            metadata: [
                'alt_text'  => $asset->metadata['alt_text'] ?? null,
                'mime_type' => $asset->mime_type,
                'file_name' => $asset->original_name,
            ],
            width: $asset->width,
            height: $asset->height,
        ));
    }

    /**
     * Sync assets in a specific collection.
     *
     * This method works like Laravel's sync method:
     * - Detaches assets not in the provided list
     * - Attaches new assets in the list
     * - Updates existing assets (e.g., order)
     * - Validates all link operations through the rules engine
     *
     * @param  string  $collection  The collection name
     * @param  array<int|Asset>  $assets  Array of asset IDs or Asset instances
     * @param  bool  $detaching  Whether to detach assets not in the list (default: true)
     * @return array{attached: array<int>, detached: array<int>, updated: array<int>}
     *
     * @throws ValidationException If any asset fails validation
     */
    public function syncAssetsInCollection(string $collection, array $assets, bool $detaching = true): array
    {
        $namespace = $this->resolveNamespace();
        $configuration = $this->getNamespaceConfiguration();
        $assetData = [];

        foreach ($assets as $index => $asset) {
            /** @var Asset * */
            $assetInstance = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);

            if ($assetInstance->media_namespace_id !== $namespace->id) {
                throw new InvalidArgumentException(
                    "Asset {$assetInstance->original_name} does not belong to namespace '{$namespace->name}'"
                );
            }

            $context = RuleContext::forLink(
                asset: $assetInstance,
                namespace: $namespace,
                folder: $assetInstance->folder,
                model: $this,
            );

            $result = $configuration->validateAction($context);

            if (!$result->passed()) {
                throw ValidationException::withMessages([
                    'asset' => [$result->getFirstError()],
                ]);
            }

            $assetData[$assetInstance->id] = [
                'collection' => $collection,
                'order'      => $index,
            ];
        }

        $currentAssets = $this->assets()
            ->wherePivot('collection', $collection)
            ->pluck('assets.id')
            ->toArray();

        $changes = [
            'attached' => [],
            'detached' => [],
            'updated'  => [],
        ];

        foreach ($assetData as $assetId => $pivotData) {
            if (in_array($assetId, $currentAssets)) {
                $this->assets()
                    ->wherePivot('collection', $collection)
                    ->updateExistingPivot($assetId, $pivotData);
                $changes['updated'][] = $assetId;
            } else {
                $this->assets()->attach($assetId, $pivotData);
                $changes['attached'][] = $assetId;
            }
        }

        if ($detaching) {
            $assetsToDetach = array_diff($currentAssets, array_keys($assetData));
            if (!empty($assetsToDetach)) {
                $this->assets()
                    ->wherePivotIn('asset_id', $assetsToDetach)
                    ->wherePivot('collection', $collection)
                    ->detach();
                $changes['detached'] = $assetsToDetach;
            }
        }

        return $changes;
    }

    /**
     * Detach assets from a specific collection.
     *
     * @param  string  $collection  The collection name
     * @param  array<int|Asset>|int|Asset|null  $assets  Asset(s) to detach, or null to detach all
     * @return int The number of assets detached
     */
    public function detachAssetsFromCollection(string $collection, array|Asset|int|null $assets = null): int
    {
        $query = $this->assets()->wherePivot('collection', $collection);

        if ($assets !== null) {
            $assetIds = [];
            $assets = is_array($assets) ? $assets : [$assets];

            foreach ($assets as $asset) {
                $assetIds[] = $asset instanceof Asset ? $asset->id : $asset;
            }

            $query->wherePivotIn('asset_id', $assetIds);
        }

        return $query->detach();
    }

    /**
     * Clear all assets from a specific collection.
     *
     * @param  string  $collection  The collection name
     * @return int The number of assets removed
     */
    public function clearAssetsInCollection(string $collection): int
    {
        return $this->detachAssetsFromCollection($collection);
    }

    /**
     * Move an asset from one collection to another.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to move
     * @param  string  $fromCollection  Source collection name
     * @param  string  $toCollection  Target collection name
     * @param  int|null  $order  Optional order in the target collection
     * @return bool True if the asset was moved, false if not found in source collection
     *
     * @throws ValidationException If the move operation fails validation
     */
    public function moveAssetToCollection(Asset|int $asset, string $fromCollection, string $toCollection, int|null $order = null): bool
    {
        $assetId = $asset instanceof Asset ? $asset->id : $asset;
        $assetInstance = $asset instanceof Asset ? $asset : Asset::findOrFail($assetId);

        $existingAsset = $this->assets()
            ->wherePivot('collection', $fromCollection)
            ->find($assetId);

        if (!$existingAsset) {
            return false;
        }

        $namespace = $this->resolveNamespace();
        $configuration = $this->getNamespaceConfiguration();

        $context = RuleContext::forLink(
            asset: $assetInstance,
            namespace: $namespace,
            folder: $assetInstance->folder,
            model: $this,
        );

        $result = $configuration->validateAction($context);

        if (!$result->passed()) {
            throw ValidationException::withMessages([
                'file' => [$result->getFirstError()],
            ]);
        }

        $this->assets()->updateExistingPivot($assetId, [
            'collection' => $toCollection,
            'order'      => $order ?? $existingAsset->pivot->order,
        ]);

        return true;
    }

    /**
     * Count assets in a specific collection.
     *
     * @param  string  $collection  The collection name
     * @return int The number of assets in the collection
     */
    public function countAssetsInCollection(string $collection): int
    {
        return $this->assets()
            ->wherePivot('collection', $collection)
            ->count();
    }

    /**
     * Check if an asset exists in a specific collection.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance
     * @param  string  $collection  The collection name
     * @return bool True if the asset exists in the collection
     */
    public function hasAssetInCollection(Asset|int $asset, string $collection): bool
    {
        $assetId = $asset instanceof Asset ? $asset->id : $asset;

        return $this->assets()
            ->wherePivot('collection', $collection)
            ->where('assets.id', $assetId)
            ->exists();
    }

    /**
     * Get collections that have assets.
     *
     * @return \Illuminate\Support\Collection<int, string> Collection names
     */
    public function getAssetCollections(): \Illuminate\Support\Collection
    {
        return $this->assets()
            ->pluck('collection')
            ->filter()
            ->unique()
            ->values();
    }

    /**
     * Duplicate an asset and attach it to a collection.
     *
     * @param  int|Asset  $asset  Asset ID or Asset instance to duplicate
     * @param  string  $collection  Collection to attach the duplicate to
     * @param  int  $order  Order position for the duplicate
     * @param  array  $metadata  Additional metadata for the duplicate
     * @return Asset The duplicated asset
     *
     * @throws ValidationException If the duplication fails validation
     */
    public function duplicateAssetToCollection(Asset|int $asset, string $collection, int $order = 0, array $metadata = []): Asset
    {
        $originalAsset = $asset instanceof Asset ? $asset : Asset::findOrFail($asset);
        $namespace = $this->resolveNamespace();
        $configuration = $this->getNamespaceConfiguration();

        if ($originalAsset->media_namespace_id !== $namespace->id) {
            throw new InvalidArgumentException(
                "Asset does not belong to namespace '{$namespace->name}'"
            );
        }

        $originalMedia = $originalAsset->getFirstMediaItem();

        if (!$originalMedia) {
            throw new InvalidArgumentException('Original asset has no media file');
        }

        $tempFile = tmpfile();
        $stream = $originalMedia->stream();
        stream_copy_to_stream($stream, $tempFile);
        rewind($tempFile);
        $tempPath = stream_get_meta_data($tempFile)['uri'];

        $uploadedFile = new UploadedFile(
            $tempPath,
            $originalMedia->file_name,
            $originalMedia->mime_type,
            null,
            true
        );

        $context = RuleContext::forUpload(
            file: $uploadedFile,
            namespace: $namespace,
            folder: $originalAsset->folder,
            model: $this,
        );

        $result = $configuration->validateAction($context);

        if (!$result->passed()) {
            fclose($tempFile);
            throw new ValidationException($result->getFirstError());
        }

        fclose($tempFile);

        return DB::transaction(function () use ($originalAsset, $collection, $order, $metadata, $originalMedia) {
            $newMetadata = array_merge(
                $originalAsset->metadata ?? [],
                [
                    'duplicated_from' => $originalAsset->id,
                    'duplicated_at'   => now()->toIso8601String(),
                    'duplicated_by'   => auth()->id(),
                ],
                $metadata
            );

            /** @var Asset */
            $newAsset = Asset::create([
                'media_namespace_id' => $originalAsset->media_namespace_id,
                'asset_folder_id'    => $originalAsset->asset_folder_id,
                'metadata'           => $newMetadata,
            ]);

            $newAsset->copyMedia($originalMedia->getPath())
                ->usingName($originalMedia->name)
                ->usingFileName($originalMedia->file_name)
                ->toMediaCollection($collection, $originalAsset->namespace->disk);

            $this->assets()->attach($newAsset->id, [
                'collection' => $collection,
                'order'      => $order,
            ]);

            return $newAsset;
        });
    }
}
